<?php
/**
 * Clase para manejar productos en Factunexo
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Product {
    
    private $api;
    
    public function __construct() {
        $this->api = new Factunexo_API();
    }
    
    /**
     * Obtener o crear producto en Factunexo
     */
    public function get_or_create_product($product) {
        if (!$product) {
            return new WP_Error('invalid_product', 'Producto inválido');
        }
        
        // Obtener código del producto
        $product_code = $this->get_product_code($product);
        
        if (empty($product_code)) {
            return new WP_Error('missing_product_code', 'El producto no tiene código');
        }
        
        // Buscar producto existente
        $existing_product = $this->find_product($product_code);
        
        if ($existing_product) {
            return $existing_product;
        }
        
        // Crear nuevo producto si está habilitado
        if (get_option('factunexo_wc_auto_create_items', 'yes') === 'yes') {
            return $this->create_product($product);
        }
        
        return new WP_Error('product_not_found', 'Producto no encontrado y la creación automática está deshabilitada');
    }
    
    /**
     * Obtener código del producto
     */
    private function get_product_code($product) {
        // Intentar obtener SKU primero
        $sku = $product->get_sku();
        if (!empty($sku)) {
            return $sku;
        }
        
        // Si no hay SKU, usar ID del producto
        return 'WC_' . $product->get_id();
    }
    
    /**
     * Buscar producto existente
     */
    private function find_product($product_code) {
        // Por ahora, retornamos null ya que la API no tiene endpoint de búsqueda de productos
        // En el futuro, esto podría implementarse si la API lo permite
        return null;
    }
    
    /**
     * Crear nuevo producto
     */
    private function create_product($product) {
        $product_data = array(
            'code' => $this->get_product_code($product),
            'name' => $product->get_name(),
            'description' => $product->get_description(),
            'price' => $product->get_price(),
            'taxRate' => $this->get_product_tax_rate($product),
            'category' => $this->get_product_category($product),
            'active' => $product->is_in_stock()
        );
        
        // Por ahora, retornamos los datos del producto ya que la API no tiene endpoint de creación
        // En el futuro, esto podría implementarse si la API lo permite
        return array(
            'id' => 'temp_' . time(),
            'code' => $product_data['code'],
            'name' => $product_data['name'],
            'description' => $product_data['description'],
            'price' => $product_data['price'],
            'taxRate' => $product_data['taxRate'],
            'category' => $product_data['category'],
            'active' => $product_data['active']
        );
    }
    
    /**
     * Obtener tasa de impuesto del producto
     */
    private function get_product_tax_rate($product) {
        // Obtener tasas de impuesto del producto
        $tax_rates = WC_Tax::get_rates($product->get_tax_class());
        
        if (!empty($tax_rates)) {
            foreach ($tax_rates as $tax_rate) {
                if (isset($tax_rate['rate'])) {
                    return $tax_rate['rate'];
                }
            }
        }
        
        return 0; // Sin IVA
    }
    
    /**
     * Obtener categoría del producto
     */
    private function get_product_category($product) {
        $categories = get_the_terms($product->get_id(), 'product_cat');
        
        if ($categories && !is_wp_error($categories)) {
            $category_names = array();
            foreach ($categories as $category) {
                $category_names[] = $category->name;
            }
            return implode(', ', $category_names);
        }
        
        return 'General';
    }
    
    /**
     * Validar datos del producto
     */
    public function validate_product_data($product_data) {
        $errors = array();
        
        if (empty($product_data['code'])) {
            $errors[] = 'El código del producto es requerido';
        }
        
        if (empty($product_data['name'])) {
            $errors[] = 'El nombre del producto es requerido';
        }
        
        if (!isset($product_data['price']) || $product_data['price'] < 0) {
            $errors[] = 'El precio del producto debe ser mayor o igual a 0';
        }
        
        if (isset($product_data['taxRate']) && $product_data['taxRate'] < 0) {
            $errors[] = 'La tasa de impuesto no puede ser negativa';
        }
        
        return $errors;
    }
    
    /**
     * Actualizar producto existente
     */
    public function update_product($product_id, $product_data) {
        // Por ahora, retornamos error ya que la API no tiene endpoint de actualización
        // En el futuro, esto podría implementarse si la API lo permite
        return new WP_Error('not_implemented', 'La actualización de productos no está implementada');
    }
    
    /**
     * Eliminar producto
     */
    public function delete_product($product_id) {
        // Por ahora, retornamos error ya que la API no tiene endpoint de eliminación
        // En el futuro, esto podría implementarse si la API lo permite
        return new WP_Error('not_implemented', 'La eliminación de productos no está implementada');
    }
}
