<?php
/**
 * Clase para manejar la visualización de pedidos con información de Factunexo
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Orders {
    
    public function __construct() {
        // Usar hooks específicos de WooCommerce
        add_filter('manage_woocommerce_page_wc-orders_columns', array($this, 'add_invoice_column'));
        add_action('manage_woocommerce_page_wc-orders_custom_column', array($this, 'display_invoice_column_content'), 10, 2);
        add_filter('manage_woocommerce_page_wc-orders_sortable_columns', array($this, 'make_invoice_column_sortable'));
        
        // También mantener los hooks tradicionales para compatibilidad
        add_filter('manage_edit-shop_order_columns', array($this, 'add_invoice_column'));
        add_action('manage_shop_order_posts_custom_column', array($this, 'display_invoice_column_content'), 10, 2);
        add_filter('manage_edit-shop_order_sortable_columns', array($this, 'make_invoice_column_sortable'));
        
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
    }
    
    /**
     * Agregar columna de factura a la tabla de pedidos
     */
    public function add_invoice_column($columns) {
        // Agregar columna de factura al final
        $columns['factunexo_invoice'] = __('Factura Factunexo', 'factunexo-woocommerce');
        return $columns;
    }
    
    /**
     * Mostrar contenido de la columna de factura
     */
    public function display_invoice_column_content($column, $post_id) {
        if ($column === 'factunexo_invoice') {
            $order = wc_get_order($post_id);
            
            if (!$order) {
                echo '—';
                return;
            }
            
            // Obtener metadatos de la factura
            $full_sequential = $order->get_meta('_factunexo_full_sequential');
            $invoice_id = $order->get_meta('_factunexo_invoice_id');
            $receipt_status = $order->get_meta('_factunexo_receipt_status');
            $emission_date = $order->get_meta('_factunexo_emission_date');
            
            if ($full_sequential) {
                // Mostrar el número de factura con estilo
                echo '<div class="factunexo-invoice-info">';
                echo '<strong style="color: #0073aa;">' . esc_html($full_sequential) . '</strong>';
                
                // Mostrar estado si está disponible
                if ($receipt_status) {
                    $status_class = $this->get_status_class($receipt_status);
                    echo '<br><small class="' . $status_class . '">' . esc_html($this->get_status_label($receipt_status)) . '</small>';
                }
                
                // Mostrar fecha de emisión si está disponible
                if ($emission_date) {
                    echo '<br><small style="color: #666;">' . esc_html($emission_date) . '</small>';
                }
                
                echo '</div>';
            } else {
                // Mostrar estado de procesamiento
                $invoice_status = $order->get_meta('_factunexo_invoice_status');
                if ($invoice_status === 'created') {
                    echo '<span style="color: #46b450;">✓ Creada</span>';
                } elseif ($invoice_status === 'error') {
                    echo '<span style="color: #dc3232;">✗ Error</span>';
                } else {
                    echo '<span style="color: #666;">—</span>';
                }
            }
        }
    }
    
    /**
     * Hacer la columna de factura ordenable
     */
    public function make_invoice_column_sortable($columns) {
        $columns['factunexo_invoice'] = 'factunexo_invoice';
        return $columns;
    }
    
    /**
     * Cargar estilos CSS para la columna de factura
     */
    public function enqueue_admin_styles($hook) {
        // Solo cargar en la página de pedidos
        if ($hook !== 'edit.php' || !isset($_GET['post_type']) || $_GET['post_type'] !== 'shop_order') {
            return;
        }
        
        wp_enqueue_style(
            'factunexo-orders',
            FACTUNEXO_WC_PLUGIN_URL . 'assets/css/orders.css',
            array(),
            FACTUNEXO_WC_VERSION
        );
    }
    
    /**
     * Obtener clase CSS para el estado
     */
    private function get_status_class($status) {
        switch ($status) {
            case 'SAVED':
                return 'status-saved';
            case 'AUTHORIZED':
                return 'status-authorized';
            case 'REJECTED':
                return 'status-rejected';
            case 'PENDING':
                return 'status-pending';
            default:
                return 'status-unknown';
        }
    }
    
    /**
     * Obtener etiqueta legible para el estado
     */
    private function get_status_label($status) {
        switch ($status) {
            case 'SAVED':
                return 'Guardada';
            case 'AUTHORIZED':
                return 'Autorizada';
            case 'REJECTED':
                return 'Rechazada';
            case 'PENDING':
                return 'Pendiente';
            default:
                return ucfirst(strtolower($status));
        }
    }
}
