<?php
/**
 * Clase para manejar los pedidos de WooCommerce
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Order_Handler {
    
    public function __construct() {
        // Solo hooks básicos y seguros que funcionan con HPOS
        add_action('woocommerce_order_status_changed', array($this, 'handle_order_status_change'), 10, 4);
    }
    
    /**
     * Obtener metadatos del pedido
     */
    private function get_order_meta($order, $key, $single = true) {
        return $order->get_meta($key, $single);
    }
    
    /**
     * Actualizar metadatos del pedido
     */
    private function update_order_meta($order, $key, $value) {
        $order->update_meta_data($key, $value);
        $order->save();
    }
    
    /**
     * Manejar cambio de estado del pedido
     */
    public function handle_order_status_change($order_id, $old_status, $new_status, $order) {
        $invoice_trigger = get_option('factunexo_wc_invoice_trigger', 'processing');
        
        // Verificar si debe emitir factura
        if (($invoice_trigger === 'processing' && $new_status === 'processing') ||
            ($invoice_trigger === 'completed' && $new_status === 'completed')) {
            
            // Verificar que no se haya emitido ya una factura
            if (!$this->has_invoice($order_id)) {
                $this->create_invoice_for_order($order);
            }
        }
    }
    
    /**
     * Verificar si el pedido ya tiene factura
     */
    private function has_invoice($order_id) {
        $order = wc_get_order($order_id);
        return !empty($this->get_order_meta($order, '_factunexo_invoice_id'));
    }
    
    /**
     * Crear factura para un pedido
     */
    private function create_invoice_for_order($order) {
        try {
            // Verificar que el pedido sea de Ecuador antes de intentar crear la factura
            if ($order->get_billing_country() !== 'EC') {
                $this->add_order_note($order, 'No se puede crear factura: El pedido no es de Ecuador (País: ' . $order->get_billing_country() . ')', 'info');
                return;
            }
            
            // Verificar que la API esté configurada
            $api = new Factunexo_API();
            if (!$api->is_configured()) {
                $this->add_order_note($order, 'No se puede crear factura: La API de Factunexo no está configurada. Ve a WooCommerce > Factunexo para configurar las credenciales.', 'error');
                return;
            }
            
            // Verificar que el punto de venta esté configurado
            $pos_id = get_option('factunexo_wc_pos_id');
            if (empty($pos_id)) {
                $this->add_order_note($order, 'No se puede crear factura: No se ha configurado un punto de venta. Ve a WooCommerce > Factunexo para seleccionar un punto de venta.', 'error');
                return;
            }
            
            $invoice_service = new Factunexo_Invoice();
            $result = $invoice_service->create_from_order($order);
            
            if (is_wp_error($result)) {
                $error_message = $result->get_error_message();
                $error_code = $result->get_error_code();
                $error_data = $result->get_error_data();
                
                // Log detallado del error
                $this->log_error('Error al crear factura para pedido ' . $order->get_id() . ': ' . $error_message);
                if ($error_data) {
                    $this->log_error('Error data: ' . print_r($error_data, true));
                }
                
                // Crear mensaje de error más detallado
                $detailed_message = 'Error al crear factura: ' . $error_message;
                
                // Agregar información específica según el tipo de error
                switch ($error_code) {
                    case 'missing_identification':
                        $detailed_message .= "\n\n🔍 **Solución:**";
                        $detailed_message .= "\n1. Verifica que el cliente completó los campos de identificación en el checkout";
                        $detailed_message .= "\n2. Asegúrate de que el país de facturación sea Ecuador";
                        $detailed_message .= "\n3. Los campos requeridos son: Tipo de identificación y Número de identificación";
                        break;
                    case 'api_not_configured':
                        $detailed_message .= "\n\n🔍 **Solución:** Ve a WooCommerce > Factunexo y configura tu API Key y Secret";
                        break;
                    case 'pos_not_configured':
                        $detailed_message .= "\n\n🔍 **Solución:** Ve a WooCommerce > Factunexo y selecciona un punto de venta";
                        break;
                    case 'invalid_country':
                        $detailed_message .= "\n\n🔍 **Información:** Solo se pueden emitir facturas para pedidos de Ecuador";
                        break;
                }
                
                $this->add_order_note($order, $detailed_message, 'error');
            } else {
                // Mostrar fullSequential si está disponible, sino invoice_id
                $invoice_number = $result['full_sequential'] ?? $result['invoice_id'];
                $this->add_order_note($order, 'Factura creada exitosamente. Número: ' . $invoice_number, 'success');
                $this->update_order_meta($order, '_factunexo_invoice_id', $result['invoice_id']);
                $this->update_order_meta($order, '_factunexo_invoice_status', 'created');
            }
        } catch (Exception $e) {
            $this->log_error('Excepción al crear factura para pedido ' . $order->get_id() . ': ' . $e->getMessage());
            $this->add_order_note($order, 'Error al crear factura: ' . $e->getMessage(), 'error');
        }
    }
    

    

    

    

    
    /**
     * Agregar nota al pedido
     */
    private function add_order_note($order, $note, $type = 'info') {
        $order->add_order_note($note, false, $type === 'error');
    }
    
    /**
     * Registrar error
     */
    private function log_error($message) {
        if (function_exists('wc_get_logger')) {
            $logger = wc_get_logger();
            $logger->error($message, array('source' => 'factunexo'));
        }
    }
}


