<?php
/**
 * Clase para manejar la creación y procesamiento de facturas
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Invoice {
    
    private $api;
    private $customer_service;
    private $product_service;
    
    public function __construct() {
        $this->api = new Factunexo_API();
        $this->customer_service = new Factunexo_Customer();
        $this->product_service = new Factunexo_Product();
    }
    
    /**
     * Crear factura desde un pedido de WooCommerce
     */
    public function create_from_order($order) {
        try {
            // Verificar que la API esté configurada
            if (!$this->api->is_configured()) {
                return new WP_Error('api_not_configured', 'La API de Factunexo no está configurada');
            }
            
            // Verificar que el pedido sea de Ecuador
            if ($order->get_billing_country() !== 'EC') {
                return new WP_Error('invalid_country', 'Solo se pueden emitir facturas para pedidos de Ecuador');
            }
            
            // Verificar que el punto de venta esté configurado
            $pos_id = get_option('factunexo_wc_pos_id');
            if (empty($pos_id)) {
                return new WP_Error('pos_not_configured', 'No se ha configurado un punto de venta. Ve a la configuración de Factunexo y selecciona un punto de venta.');
            }
            
            // Obtener o crear cliente
            $customer_data = $this->customer_service->get_or_create_customer($order);
            if (is_wp_error($customer_data)) {
                // Log detallado del error
                error_log('Factunexo Invoice Creation Failed - Order ID: ' . $order->get_id());
                error_log('Factunexo Invoice Creation Failed - Error: ' . $customer_data->get_error_message());
                error_log('Factunexo Invoice Creation Failed - Error Data: ' . print_r($customer_data->get_error_data(), true));
                return $customer_data;
            }
            
            // Preparar datos de la factura
            $invoice_data = $this->prepare_invoice_data($order, $customer_data);
            
            // Guardar JSON en campo personalizado de la orden para debugging
            $order->update_meta_data('_factunexo_invoice_json', json_encode($invoice_data, JSON_PRETTY_PRINT));
            $order->save();
            
            // Log del JSON que se va a enviar
            error_log('Factunexo - JSON a enviar: ' . json_encode($invoice_data, JSON_PRETTY_PRINT));
            
            // Obtener merchantId de la configuración
            $merchant_id = get_option('factunexo_wc_merchant_id');
            
            // Log del merchant ID
            error_log('Factunexo - Merchant ID: ' . $merchant_id);
            
            // Crear factura en Factunexo
            $result = $this->api->create_invoice($invoice_data, $merchant_id);
            
            if (is_wp_error($result)) {
                return $result;
            }
            
            // Guardar metadatos de la respuesta en la orden
            $this->save_invoice_metadata($order, $result);
            
            return array(
                'invoice_id' => $result['invoiceId'] ?? $result['id'],
                'receipt_id' => $result['receipt']['receiptId'] ?? null,
                'full_sequential' => $result['receipt']['fullSequential'] ?? null,
                'status' => 'created',
                'message' => 'Factura creada exitosamente'
            );
            
        } catch (Exception $e) {
            return new WP_Error('invoice_creation_error', 'Error al crear factura: ' . $e->getMessage());
        }
    }
    
    /**
     * Preparar datos de la factura para el nuevo endpoint público
     */
    private function prepare_invoice_data($order, $customer_data) {
        $pos_id = get_option('factunexo_wc_pos_id');
        $items = array();
        
        // Procesar cada item del pedido
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $product_data = $this->product_service->get_or_create_product($product);
            
            if (is_wp_error($product_data)) {
                continue; // Saltar productos con error
            }
            
            $items[] = array(
                'code' => $product_data['code'],
                'description' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'price' => $item->get_total() / $item->get_quantity(),
                'discountType' => 'PERCENTAGE', // Tipo de descuento por defecto
                'discount' => 0,
                'discountPercentage' => 0,
                'totalWithoutTaxes' => $item->get_subtotal()
            );
        }
        
        // Log de los datos de la orden
        error_log('Factunexo - Datos de la orden:');
        error_log('- _wc_billing/factunexo/identification-type: ' . $order->get_meta('_wc_billing/factunexo/identification-type'));
        error_log('- _wc_billing/factunexo/identification-number: ' . $order->get_meta('_wc_billing/factunexo/identification-number'));
        error_log('- _billing_identification_type: ' . $order->get_meta('_billing_identification_type'));
        error_log('- _billing_identification_number: ' . $order->get_meta('_billing_identification_number'));
        error_log('- _billing_use_company: ' . $order->get_meta('_billing_use_company'));
        error_log('- _billing_company_name: ' . $order->get_meta('_billing_company_name'));
        error_log('- billing_first_name: ' . $order->get_billing_first_name());
        error_log('- billing_last_name: ' . $order->get_billing_last_name());
        error_log('- billing_email: ' . $order->get_billing_email());
        error_log('- billing_phone: ' . $order->get_billing_phone());
        
        // Determinar nombre del cliente
        $customer_name = $order->get_meta('_billing_use_company') && $order->get_meta('_billing_company_name') 
            ? $order->get_meta('_billing_company_name')
            : $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        
        // Estructura para el nuevo endpoint público
        $invoice_data = array(
            'receipt' => array(
                'pos' => array(
                    'posId' => $pos_id
                ),
                'customer' => array(
                    'identificationType' => $this->map_identification_type($this->get_identification_type($order)),
                    'identification' => $this->get_valid_identification($order),
                    'taxpayerName' => $customer_name,
                    'email' => $order->get_billing_email(),
                    'phoneNumber' => $order->get_billing_phone(),
                    'address' => $order->get_billing_address_1() . ', ' . $order->get_billing_city()
                )
            ),
            'invoiceInfo' => array(
                'posId' => $pos_id,
                'subtotal' => $order->get_subtotal(),
                'taxAmount' => $order->get_total_tax(),
                'totalAmount' => $order->get_total(),
                'currency' => $order->get_currency(),
                'notes' => 'Pedido WooCommerce #' . $order->get_order_number(),
                'externalReference' => $order->get_id()
            ),
            'invoiceItems' => $items,
            'additionalFields' => array()
        );
        
        // Log del mapeo de identificación
        $original_type = $this->get_identification_type($order);
        $mapped_type = $this->map_identification_type($original_type);
        $valid_identification = $this->get_valid_identification($order);
        error_log('Factunexo - Mapeo de identificación:');
        error_log('- Tipo original: ' . $original_type);
        error_log('- Tipo mapeado: ' . $mapped_type);
        error_log('- Identificación: ' . $valid_identification);
        
        return $invoice_data;
    }
    
    /**
     * Mapear tipo de identificación del plugin al enum del API
     */
    private function map_identification_type($plugin_type) {
        $mapping = array(
            'cedula' => 'IDENTIFICATION_CARD',
            'ruc' => 'RUC',
            'pasaporte' => 'PASSPORT',
            'extranjero' => 'FOREIGN_IDENTIFICATION',
            'consumidor_final' => 'END_CUSTOMER'
        );
        
        $normalized_type = strtolower($plugin_type ?: 'cedula');
        return $mapping[$normalized_type] ?? 'IDENTIFICATION_CARD';
    }
    
    /**
     * Guardar metadatos de la factura en la orden
     */
    private function save_invoice_metadata($order, $result) {
        // Guardar invoiceId
        if (isset($result['invoiceId'])) {
            $order->update_meta_data('_factunexo_invoice_id', $result['invoiceId']);
        }
        
        // Guardar receiptId
        if (isset($result['receipt']['receiptId'])) {
            $order->update_meta_data('_factunexo_receipt_id', $result['receipt']['receiptId']);
        }
        
        // Guardar fullSequential
        if (isset($result['receipt']['fullSequential'])) {
            $order->update_meta_data('_factunexo_full_sequential', $result['receipt']['fullSequential']);
        }
        
        // Guardar estado del recibo
        if (isset($result['receipt']['receiptStatus'])) {
            $order->update_meta_data('_factunexo_receipt_status', $result['receipt']['receiptStatus']);
        }
        
        // Guardar fecha de emisión
        if (isset($result['receipt']['emissionDate'])) {
            $order->update_meta_data('_factunexo_emission_date', $result['receipt']['emissionDate']);
        }
        
        // Guardar número de autorización
        if (isset($result['receipt']['authorizationNumber'])) {
            $order->update_meta_data('_factunexo_authorization_number', $result['receipt']['authorizationNumber']);
        }
        
        // Guardar respuesta completa para debugging
        $order->update_meta_data('_factunexo_invoice_response', json_encode($result, JSON_PRETTY_PRINT));
        
        $order->save();
        
        // Log de los metadatos guardados
        error_log('Factunexo - Metadatos guardados:');
        error_log('- Invoice ID: ' . ($result['invoiceId'] ?? 'N/A'));
        error_log('- Receipt ID: ' . ($result['receipt']['receiptId'] ?? 'N/A'));
        error_log('- Full Sequential: ' . ($result['receipt']['fullSequential'] ?? 'N/A'));
    }
    
    /**
     * Obtener tipo de identificación del pedido
     */
    private function get_identification_type($order) {
        // Buscar en los campos de Factunexo primero
        $type = $order->get_meta('_wc_billing/factunexo/identification-type');
        
        // Si no está en los campos de Factunexo, buscar en los campos tradicionales
        if (empty($type)) {
            $type = $order->get_meta('_billing_identification_type');
        }
        
        return $type;
    }
    
    /**
     * Obtener identificación válida del pedido
     */
    private function get_valid_identification($order) {
        // Buscar en los campos de Factunexo primero
        $identification = $order->get_meta('_wc_billing/factunexo/identification-number');
        
        // Si no está en los campos de Factunexo, buscar en los campos tradicionales
        if (empty($identification)) {
            $identification = $order->get_meta('_billing_identification_number');
        }
        
        // Si no hay identificación, devolver vacío (no usar consumidor final por ahora)
        if (empty($identification)) {
            return '';
        }
        
        return $identification;
    }
    
    /**
     * Obtener tasa de impuesto para un item
     */
    private function get_tax_rate($item) {
        // Obtener la tasa de impuesto del item
        $tax_total = $item->get_total_tax();
        $subtotal = $item->get_subtotal();
        
        if ($tax_total > 0 && $subtotal > 0) {
            // Calcular la tasa de impuesto como porcentaje
            $tax_rate = ($tax_total / $subtotal) * 100;
            return round($tax_rate, 2);
        }
        
        // Si no hay impuestos, intentar obtener la tasa por defecto de Ecuador (12% IVA)
        $country = $item->get_order()->get_billing_country();
        if ($country === 'EC') {
            return 12.00; // IVA estándar en Ecuador
        }
        
        return 0; // Sin IVA
    }
    
    /**
     * Procesar factura existente
     */
    public function process_invoice($invoice_id) {
        if (!$this->api->is_configured()) {
            return new WP_Error('api_not_configured', 'La API de Factunexo no está configurada');
        }
        
        $result = $this->api->process_invoice($invoice_id);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return array(
            'invoice_id' => $invoice_id,
            'status' => 'processed',
            'message' => 'Factura procesada exitosamente'
        );
    }
    
    /**
     * Obtener información de una factura
     */
    public function get_invoice($invoice_id) {
        if (!$this->api->is_configured()) {
            return new WP_Error('api_not_configured', 'La API de Factunexo no está configurada');
        }
        
        return $this->api->get_invoice($invoice_id);
    }
    
    /**
     * Actualizar factura
     */
    public function update_invoice($invoice_id, $invoice_data) {
        if (!$this->api->is_configured()) {
            return new WP_Error('api_not_configured', 'La API de Factunexo no está configurada');
        }
        
        return $this->api->update_invoice($invoice_id, $invoice_data);
    }
    
    /**
     * Guardar y procesar factura
     */
    public function save_and_process_invoice($invoice_id, $invoice_data) {
        if (!$this->api->is_configured()) {
            return new WP_Error('api_not_configured', 'La API de Factunexo no está configurada');
        }
        
        return $this->api->save_and_process_invoice($invoice_id, $invoice_data);
    }
}
