<?php
/**
 * Clase para manejar clientes en Factunexo
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Customer {
    
    private $api;
    
    public function __construct() {
        $this->api = new Factunexo_API();
    }
    
    /**
     * Obtener o crear cliente en Factunexo
     * Usa el helper oficial de WooCommerce para leer campos adicionales
     */
    public function get_or_create_customer($order) {
        // Verificar que el pedido sea de Ecuador
        if ($order->get_billing_country() !== 'EC') {
            return new WP_Error('invalid_country', 'Solo se pueden emitir facturas para pedidos de Ecuador');
        }
        
        // Usar el helper de WooCommerce para obtener los campos adicionales
        $identification_type = $this->get_additional_field($order, 'factunexo/identification-type', 'billing');
        $identification_number = $this->get_additional_field($order, 'factunexo/identification-number', 'billing');
        
        // Verificar datos de identificación con información detallada
        $missing_fields = array();
        if (empty($identification_type)) {
            $missing_fields[] = 'tipo de identificación';
        }
        if (empty($identification_number)) {
            $missing_fields[] = 'número de identificación';
        }
        
        if (!empty($missing_fields)) {
            $error_message = 'Faltan datos de identificación del cliente: ' . implode(', ', $missing_fields);
            $error_message .= '. Asegúrate de que el cliente complete los campos de identificación en el checkout.';
            
            // Log detallado para depuración
            error_log('Factunexo Customer Error - Order ID: ' . $order->get_id());
            error_log('Factunexo Customer Error - Billing Country: ' . $order->get_billing_country());
            error_log('Factunexo Customer Error - Identification Type: ' . ($identification_type ?: 'EMPTY'));
            error_log('Factunexo Customer Error - Identification Number: ' . ($identification_number ?: 'EMPTY'));
            error_log('Factunexo Customer Error - All Additional Fields: ' . print_r($this->get_all_additional_fields($order), true));
            
            return new WP_Error('missing_identification', $error_message, array(
                'missing_fields' => $missing_fields,
                'order_id' => $order->get_id(),
                'billing_country' => $order->get_billing_country()
            ));
        }
        
        // Buscar cliente existente
        $existing_customer = $this->find_customer($identification_type, $identification_number);
        
        if ($existing_customer) {
            return $existing_customer;
        }
        
        // Crear nuevo cliente si está habilitado
        if (get_option('factunexo_wc_auto_create_customers', 'yes') === 'yes') {
            return $this->create_customer($order);
        }
        
        return new WP_Error('customer_not_found', 'Cliente no encontrado y la creación automática está deshabilitada');
    }
    
    /**
     * Buscar cliente existente
     */
    private function find_customer($identification_type, $identification_number) {
        // Por ahora, retornamos null ya que la API no tiene endpoint de búsqueda de clientes
        // En el futuro, esto podría implementarse si la API lo permite
        return null;
    }
    
    /**
     * Crear nuevo cliente
     */
    private function create_customer($order) {
        $customer_data = array(
            'identificationType' => $this->get_additional_field($order, 'factunexo/identification-type', 'billing'),
            'identificationNumber' => $this->get_additional_field($order, 'factunexo/identification-number', 'billing'),
            'name' => $this->get_customer_name($order),
            'email' => $order->get_billing_email(),
            'phone' => $order->get_billing_phone(),
            'address' => array(
                'street' => $order->get_billing_address_1(),
                'city' => $order->get_billing_city(),
                'state' => $order->get_billing_state(),
                'country' => $order->get_billing_country(),
                'postalCode' => $order->get_billing_postcode()
            )
        );
        
        // Por ahora, retornamos los datos del cliente ya que la API no tiene endpoint de creación
        // En el futuro, esto podría implementarse si la API lo permite
        return array(
            'id' => 'temp_' . time(),
            'identificationType' => $customer_data['identificationType'],
            'identificationNumber' => $customer_data['identificationNumber'],
            'name' => $customer_data['name'],
            'email' => $customer_data['email'],
            'phone' => $customer_data['phone'],
            'address' => $customer_data['address']
        );
    }
    
    /**
     * Obtener nombre del cliente
     */
    private function get_customer_name($order) {
        $use_company = $this->get_additional_field($order, 'factunexo/use-company', 'other');
        $company_name = $this->get_additional_field($order, 'factunexo/company-name', 'other');
        
        if ($use_company && !empty($company_name)) {
            return $company_name;
        }
        
        $first_name = $order->get_billing_first_name();
        $last_name = $order->get_billing_last_name();
        
        return trim($first_name . ' ' . $last_name);
    }
    
    /**
     * Obtener un campo adicional usando el helper de WooCommerce
     * @see https://developer.woocommerce.com/docs/block-development/extensible-blocks/cart-and-checkout-blocks/additional-checkout-fields/
     */
    private function get_additional_field($order, $field_id, $group = 'billing') {
        // Verificar si el helper de WooCommerce está disponible
        if (!class_exists('\Automattic\WooCommerce\Blocks\Package') || 
            !class_exists('\Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields')) {
            // Fallback: intentar leer del meta directo
            return $this->get_field_from_meta($order, $field_id, $group);
        }
        
        try {
            $checkout_fields = \Automattic\WooCommerce\Blocks\Package::container()->get(
                \Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields::class
            );
            
            return $checkout_fields->get_field_from_object($field_id, $order, $group);
        } catch (Exception $e) {
            error_log('Factunexo: Error al obtener campo adicional: ' . $e->getMessage());
            // Fallback: intentar leer del meta directo
            return $this->get_field_from_meta($order, $field_id, $group);
        }
    }
    
    /**
     * Obtener todos los campos adicionales del pedido
     */
    private function get_all_additional_fields($order) {
        if (!class_exists('\Automattic\WooCommerce\Blocks\Package') || 
            !class_exists('\Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields')) {
            return array();
        }
        
        try {
            $checkout_fields = \Automattic\WooCommerce\Blocks\Package::container()->get(
                \Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields::class
            );
            
            return array(
                'billing' => $checkout_fields->get_all_fields_from_object($order, 'billing'),
                'shipping' => $checkout_fields->get_all_fields_from_object($order, 'shipping'),
                'other' => $checkout_fields->get_all_fields_from_object($order, 'other'),
            );
        } catch (Exception $e) {
            error_log('Factunexo: Error al obtener todos los campos adicionales: ' . $e->getMessage());
            return array();
        }
    }
    
    /**
     * Fallback para obtener campo directamente del meta
     * Intenta con diferentes formatos de clave
     */
    private function get_field_from_meta($order, $field_id, $group) {
        // Intentar diferentes formatos de meta key
        $meta_keys = array(
            $field_id, // factunexo/identification-type
            str_replace('/', '_', $field_id), // factunexo_identification-type
            str_replace('/', '_', str_replace('-', '_', $field_id)), // factunexo_identification_type
            '_' . str_replace('/', '_', str_replace('-', '_', $field_id)), // _factunexo_identification_type
        );
        
        foreach ($meta_keys as $meta_key) {
            $value = $order->get_meta($meta_key);
            if (!empty($value)) {
                return $value;
            }
        }
        
        return null;
    }
    
    /**
     * Validar datos del cliente
     */
    public function validate_customer_data($customer_data) {
        $errors = array();
        
        if (empty($customer_data['identificationType'])) {
            $errors[] = 'El tipo de identificación es requerido';
        }
        
        if (empty($customer_data['identificationNumber'])) {
            $errors[] = 'El número de identificación es requerido';
        }
        
        if (empty($customer_data['name'])) {
            $errors[] = 'El nombre del cliente es requerido';
        }
        
        if (empty($customer_data['email'])) {
            $errors[] = 'El email del cliente es requerido';
        } elseif (!is_email($customer_data['email'])) {
            $errors[] = 'El email del cliente no es válido';
        }
        
        return $errors;
    }
    
    /**
     * Depurar datos del cliente para un pedido
     */
    public function debug_customer_data($order) {
        $debug_info = array(
            'order_id' => $order->get_id(),
            'billing_country' => $order->get_billing_country(),
            'identification_type' => $this->get_additional_field($order, 'factunexo/identification-type', 'billing'),
            'identification_number' => $this->get_additional_field($order, 'factunexo/identification-number', 'billing'),
            'company_name' => $this->get_additional_field($order, 'factunexo/company-name', 'other'),
            'use_company' => $this->get_additional_field($order, 'factunexo/use-company', 'other'),
            'billing_email' => $order->get_billing_email(),
            'billing_phone' => $order->get_billing_phone(),
            'billing_first_name' => $order->get_billing_first_name(),
            'billing_last_name' => $order->get_billing_last_name(),
            'all_additional_fields' => $this->get_all_additional_fields($order),
            'all_meta_data' => array()
        );
        
        // Obtener todos los metadatos del pedido
        $meta_data = $order->get_meta_data();
        foreach ($meta_data as $meta) {
            $debug_info['all_meta_data'][$meta->get_data()['key']] = $meta->get_data()['value'];
        }
        
        return $debug_info;
    }
}
