<?php
/**
 * Clase principal para manejar la comunicación con la API de Factunexo
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_API {
    
    private $api_key;
    private $api_secret;
    private $api_url;
    
    public function __construct() {
        $this->api_key = get_option('factunexo_wc_api_key');
        $this->api_secret = get_option('factunexo_wc_api_secret');
        $this->api_url = get_option('factunexo_wc_api_url', 'https://api.factunexo.com');
    }
    
    /**
     * Verificar si la API está configurada
     */
    public function is_configured() {
        return !empty($this->api_key) && !empty($this->api_secret);
    }
    
    /**
     * Obtener información de depuración de las credenciales
     */
    public function get_credentials_debug_info() {
        return array(
            'api_key_configured' => !empty($this->api_key),
            'api_secret_configured' => !empty($this->api_secret),
            'api_key_length' => strlen($this->api_key ?? ''),
            'api_secret_length' => strlen($this->api_secret ?? ''),
            'api_key_preview' => !empty($this->api_key) ? substr($this->api_key, 0, 8) . '...' . substr($this->api_key, -4) : 'No configurado',
            'api_secret_preview' => !empty($this->api_secret) ? substr($this->api_secret, 0, 8) . '...' . substr($this->api_secret, -4) : 'No configurado',
            'api_url' => $this->api_url
        );
    }
    
    /**
     * Realizar una petición a la API
     */
    private function make_request($endpoint, $method = 'GET', $data = null) {
        if (!$this->is_configured()) {
            return new WP_Error('not_configured', 'La API de Factunexo no está configurada');
        }
        
        $url = trailingslashit($this->api_url) . 'api/publicapi/' . ltrim($endpoint, '/');
        
        $args = array(
            'method' => $method,
            'headers' => array(
                'X-API-Key' => $this->api_key,
                'X-API-Secret' => $this->api_secret,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'timeout' => 30,
        );
        
        if ($data && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            $args['body'] = json_encode($data);
        }
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Debug logging
        error_log('Factunexo API Request: ' . $url);
        error_log('Factunexo API Response Code: ' . $response_code);
        error_log('Factunexo API Response Body: ' . $response_body);
        
        if ($response_code >= 200 && $response_code < 300) {
            $decoded = json_decode($response_body, true);
            error_log('Factunexo API Decoded: ' . print_r($decoded, true));
            return $decoded;
        } else {
            $error_data = json_decode($response_body, true);
            
            // Determinar el tipo de error basado en el código de respuesta
            $error_type = 'api_error';
            $error_message = 'Error en la API';
            
            switch ($response_code) {
                case 401:
                    $error_type = 'invalid_credentials';
                    $error_message = 'Credenciales de API inválidas';
                    break;
                case 403:
                    $error_type = 'access_denied';
                    $error_message = 'Acceso denegado a la API';
                    break;
                case 404:
                    $error_type = 'not_found';
                    $error_message = 'Endpoint no encontrado';
                    break;
                case 422:
                    $error_type = 'validation_error';
                    $error_message = 'Error de validación en los datos enviados';
                    break;
                case 500:
                    $error_type = 'server_error';
                    $error_message = 'Error interno del servidor';
                    break;
                default:
                    if (isset($error_data['error'])) {
                        $error_message = $error_data['error'];
                    } else if (isset($error_data['message'])) {
                        $error_message = $error_data['message'];
                    }
            }
            
            // Logging detallado para depuración
            error_log('Factunexo API Error Details:');
            error_log('- URL: ' . $url);
            error_log('- Method: ' . $method);
            error_log('- Response Code: ' . $response_code);
            error_log('- Response Body: ' . $response_body);
            error_log('- Error Type: ' . $error_type);
            error_log('- Error Message: ' . $error_message);
            
            // Agregar información adicional al mensaje de error
            $detailed_message = $error_message;
            if ($response_code === 401) {
                $detailed_message .= ' (Código: 401 - Verifica que tu API Key y Secret sean correctos)';
            } else if ($response_code === 403) {
                $detailed_message .= ' (Código: 403 - Tu cuenta no tiene permisos para acceder a este recurso)';
            } else if ($response_code >= 500) {
                $detailed_message .= ' (Código: ' . $response_code . ' - Error del servidor, intenta más tarde)';
            } else {
                $detailed_message .= ' (Código: ' . $response_code . ')';
            }
            
            return new WP_Error($error_type, $detailed_message, array(
                'status' => $response_code,
                'url' => $url,
                'response_body' => $response_body,
                'error_data' => $error_data
            ));
        }
    }
    
    /**
     * Obtener catálogo de POS
     */
    public function get_pos_catalog() {
        return $this->make_request('pos/catalog');
    }
    
    /**
     * Crear una factura usando el nuevo endpoint público
     */
    public function create_invoice($invoice_data, $merchant_id = null) {
        $endpoint = 'invoices';
        
        // Agregar merchantId como parámetro de consulta si está disponible
        if ($merchant_id) {
            $endpoint .= '?merchantId=' . urlencode($merchant_id);
        }
        
        return $this->make_request($endpoint, 'POST', $invoice_data);
    }
    
    /**
     * Obtener una factura por ID
     */
    public function get_invoice($invoice_id) {
        return $this->make_request("invoices/{$invoice_id}");
    }
    
    /**
     * Actualizar una factura
     */
    public function update_invoice($invoice_id, $invoice_data) {
        return $this->make_request("invoices/{$invoice_id}", 'PUT', $invoice_data);
    }
    
    /**
     * Guardar y procesar una factura
     */
    public function save_and_process_invoice($invoice_id, $invoice_data) {
        return $this->make_request("invoices/{$invoice_id}/save-and-process", 'PATCH', $invoice_data);
    }
    
    /**
     * Procesar una factura existente
     */
    public function process_invoice($invoice_id) {
        return $this->make_request("invoices/{$invoice_id}/process", 'POST');
    }
    
    /**
     * Verificar la conexión con la API
     */
    public function test_connection() {
        // Verificar configuración primero
        if (!$this->is_configured()) {
            $debug_info = $this->get_credentials_debug_info();
            return array(
                'success' => false,
                'message' => 'La API de Factunexo no está configurada. Por favor, ingresa tu API Key y Secret.',
                'error_type' => 'not_configured',
                'debug_info' => $debug_info
            );
        }
        
        // Log de información de depuración antes de la petición
        $debug_info = $this->get_credentials_debug_info();
        error_log('Factunexo Connection Test - Debug Info: ' . print_r($debug_info, true));
        
        $response = $this->get_pos_catalog();
        
        if (is_wp_error($response)) {
            $error_code = $response->get_error_code();
            $error_message = $response->get_error_message();
            $error_data = $response->get_error_data();
            
            // Información adicional para depuración
            $debug_info = array();
            if (isset($error_data['status'])) {
                $debug_info[] = 'Código HTTP: ' . $error_data['status'];
            }
            if (isset($error_data['url'])) {
                $debug_info[] = 'URL: ' . $error_data['url'];
            }
            
            $detailed_message = $error_message;
            if (!empty($debug_info)) {
                $detailed_message .= ' (' . implode(', ', $debug_info) . ')';
            }
            
            // Agregar información específica para errores 401
            if (isset($error_data['status']) && $error_data['status'] == 401) {
                $detailed_message .= "\n\n🔍 **Posibles causas del error 401:**\n";
                $detailed_message .= "1. API Key no existe en la base de datos\n";
                $detailed_message .= "2. API Key está inactiva o expirada\n";
                $detailed_message .= "3. Secret Key no coincide\n";
                $detailed_message .= "4. Problemas de conectividad con la API\n";
                $detailed_message .= "5. URL de API incorrecta\n\n";
                $detailed_message .= "**Información de depuración:**\n";
                $detailed_message .= "- API Key configurado: " . ($this->api_key ? 'Sí' : 'No') . "\n";
                $detailed_message .= "- API Secret configurado: " . ($this->api_secret ? 'Sí' : 'No') . "\n";
                $detailed_message .= "- URL de API: " . $this->api_url . "\n";
            }
            
            return array(
                'success' => false,
                'message' => $detailed_message,
                'error_type' => $error_code,
                'error_data' => $error_data,
                'debug_info' => $this->get_credentials_debug_info()
            );
        }
        
        $pos_count = count($response ?? []);
        $message = 'Conexión exitosa con la API de Factunexo';
        
        if ($pos_count > 0) {
            $message .= '. Se encontraron ' . $pos_count . ' puntos de venta disponibles.';
        } else {
            $message .= '. No se encontraron puntos de venta en tu cuenta.';
        }
        
        return array(
            'success' => true,
            'message' => $message,
            'pos_count' => $pos_count,
            'api_url' => $this->api_url,
            'debug_info' => $this->get_credentials_debug_info()
        );
    }
    
    /**
     * Probar conectividad básica con la API
     */
    public function test_basic_connectivity() {
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'message' => 'Credenciales no configuradas',
                'error_type' => 'not_configured'
            );
        }
        
        $url = trailingslashit($this->api_url) . 'api/publicapi/pos/catalog';
        
        // Realizar una petición simple para probar conectividad
        $args = array(
            'method' => 'GET',
            'timeout' => 10,
            'headers' => array(
                'X-API-Key' => $this->api_key,
                'X-API-Secret' => $this->api_secret,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
        );
        
        error_log('Factunexo Basic Connectivity Test - URL: ' . $url);
        error_log('Factunexo Basic Connectivity Test - Headers: ' . print_r($args['headers'], true));
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => 'Error de conectividad: ' . $response->get_error_message(),
                'error_type' => 'connectivity_error'
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        error_log('Factunexo Basic Connectivity Test - Response Code: ' . $response_code);
        error_log('Factunexo Basic Connectivity Test - Response Body: ' . $response_body);
        
        return array(
            'success' => $response_code >= 200 && $response_code < 300,
            'message' => 'Conectividad básica: ' . ($response_code >= 200 && $response_code < 300 ? 'OK' : 'Error'),
            'response_code' => $response_code,
            'response_body' => $response_body,
            'url' => $url
        );
    }
    
    /**
     * Obtener información de la cuenta
     */
    public function get_account_info() {
        // Esta función podría implementarse si la API expone información de la cuenta
        $pos_catalog = $this->get_pos_catalog();
        
        if (is_wp_error($pos_catalog)) {
            return $pos_catalog;
        }
        
        return array(
            'pos_available' => count($pos_catalog ?? []),
            'api_status' => 'active'
        );
    }
}
