<?php
/**
 * Clase para manejar el checkout de WooCommerce
 * Usa la API oficial de WooCommerce Additional Checkout Fields
 *
 * @package FactunexoWooCommerce
 * @see https://developer.woocommerce.com/docs/block-development/extensible-blocks/cart-and-checkout-blocks/additional-checkout-fields/
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Checkout {
    
    public function __construct() {
        // Registrar campos usando la API oficial de WooCommerce
        add_action('woocommerce_init', array($this, 'register_checkout_fields'));
        
        // Hooks de sanitización y validación
        add_action('woocommerce_sanitize_additional_field', array($this, 'sanitize_fields'), 10, 2);
        add_action('woocommerce_validate_additional_field', array($this, 'validate_fields'), 10, 3);
        add_action('woocommerce_blocks_validate_location_address_fields', array($this, 'validate_address_fields'), 10, 3);
        
        // Hooks para compatibilidad con checkout tradicional
        add_action('woocommerce_after_checkout_billing_form', array($this, 'add_fields_to_traditional_checkout'));
        add_action('woocommerce_checkout_process', array($this, 'validate_traditional_checkout'));
        
        // Scripts y estilos
        add_action('wp_enqueue_scripts', array($this, 'enqueue_checkout_scripts'));
    }
    
    /**
     * Registrar campos de checkout usando la API oficial de WooCommerce
     * Solo se muestran cuando el país es Ecuador
     */
    public function register_checkout_fields() {
        // Verificar que WooCommerce esté activo y tenga la función
        if (!function_exists('woocommerce_register_additional_checkout_field')) {
            error_log('Factunexo: woocommerce_register_additional_checkout_field no está disponible');
            return;
        }
        
        /**
         * Campo 1: Tipo de Identificación
         * Ubicación: address (aparecerá en dirección de facturación y envío)
         */
        woocommerce_register_additional_checkout_field(
            array(
                'id'            => 'factunexo/identification-type',
                'label'         => __('Tipo de Identificación', 'factunexo-woocommerce'),
                'location'      => 'address',
                'type'          => 'select',
                'required'      => true,
                'options'       => array(
                    array(
                        'value' => '',
                        'label' => __('Selecciona el tipo', 'factunexo-woocommerce'),
                    ),
                    array(
                        'value' => 'CEDULA',
                        'label' => __('Cédula de Identidad', 'factunexo-woocommerce'),
                    ),
                    array(
                        'value' => 'RUC',
                        'label' => __('RUC', 'factunexo-woocommerce'),
                    ),
                    array(
                        'value' => 'PASAPORTE',
                        'label' => __('Pasaporte', 'factunexo-woocommerce'),
                    ),
                ),
                'attributes'    => array(
                    'autocomplete' => 'off',
                ),
            )
        );
        
        /**
         * Campo 2: Número de Identificación
         * Ubicación: address (aparecerá en dirección de facturación y envío)
         */
        woocommerce_register_additional_checkout_field(
            array(
                'id'            => 'factunexo/identification-number',
                'label'         => __('Número de Identificación', 'factunexo-woocommerce'),
                'location'      => 'address',
                'type'          => 'text',
                'required'      => true,
                'attributes'    => array(
                    'autocomplete' => 'off',
                    'pattern'      => '[0-9]{10,13}',
                    'title'        => __('Debe tener entre 10 y 13 dígitos', 'factunexo-woocommerce'),
                    'maxlength'    => 13,
                    'placeholder'  => __('Ingresa tu número de identificación', 'factunexo-woocommerce'),
                ),
            )
        );
        
        /**
         * Campo 3: Nombre de la Empresa
         * Ubicación: order (información adicional del pedido)
         */
        woocommerce_register_additional_checkout_field(
            array(
                'id'            => 'factunexo/company-name',
                'label'         => __('Nombre de la Empresa', 'factunexo-woocommerce'),
                'location'      => 'order',
                'type'          => 'text',
                'required'      => false,
                'attributes'    => array(
                    'autocomplete' => 'organization',
                    'placeholder'  => __('Solo si deseas factura a nombre de empresa', 'factunexo-woocommerce'),
                ),
            )
        );
        
        /**
         * Campo 4: Checkbox para usar empresa
         * Ubicación: order (información adicional del pedido)
         */
        woocommerce_register_additional_checkout_field(
            array(
                'id'            => 'factunexo/use-company',
                'label'         => __('Emitir factura a nombre de la empresa', 'factunexo-woocommerce'),
                'location'      => 'order',
                'type'          => 'checkbox',
                'required'      => false,
            )
        );
        
        error_log('Factunexo: Campos de checkout registrados correctamente');
    }
    
    /**
     * Sanitizar campos antes de guardar
     */
    public function sanitize_fields($field_value, $field_key) {
        switch ($field_key) {
            case 'factunexo/identification-type':
                // Sanitizar y validar que sea uno de los valores permitidos
                $field_value = sanitize_text_field($field_value);
                $allowed_types = array('CEDULA', 'RUC', 'PASAPORTE');
                if (!in_array($field_value, $allowed_types)) {
                    $field_value = '';
                }
                break;
                
            case 'factunexo/identification-number':
                // Remover espacios, guiones y caracteres no numéricos
                $field_value = preg_replace('/[^0-9]/', '', $field_value);
                break;
                
            case 'factunexo/company-name':
                $field_value = sanitize_text_field($field_value);
                break;
                
            case 'factunexo/use-company':
                $field_value = (bool) $field_value;
                break;
        }
        
        return $field_value;
    }
    
    /**
     * Validar campos individuales
     */
    public function validate_fields($errors, $field_key, $field_value) {
        // Validar tipo de identificación
        if ('factunexo/identification-type' === $field_key) {
            $allowed_types = array('CEDULA', 'RUC', 'PASAPORTE');
            if (!in_array($field_value, $allowed_types)) {
                $errors->add(
                    'invalid_identification_type',
                    __('Por favor selecciona un tipo de identificación válido.', 'factunexo-woocommerce')
                );
            }
        }
        
        // Validar número de identificación
        if ('factunexo/identification-number' === $field_key) {
            // Verificar que solo contenga números
            if (!preg_match('/^[0-9]{10,13}$/', $field_value)) {
                $errors->add(
                    'invalid_identification_number',
                    __('El número de identificación debe tener entre 10 y 13 dígitos.', 'factunexo-woocommerce')
                );
            }
            
            // Validación específica según el tipo (se hará en validate_address_fields)
        }
        
        return $errors;
    }
    
    /**
     * Validar campos de dirección en conjunto
     * Aquí validamos la cédula/RUC según el tipo seleccionado
     */
    public function validate_address_fields($errors, $fields, $group) {
        // Solo validar si hay campos de Factunexo
        if (!isset($fields['factunexo/identification-type']) || !isset($fields['factunexo/identification-number'])) {
            return $errors;
        }
        
        $identification_type = $fields['factunexo/identification-type'];
        $identification_number = $fields['factunexo/identification-number'];
        
        // Validar según el tipo
        switch ($identification_type) {
            case 'CEDULA':
                if (!$this->validate_cedula($identification_number)) {
                    $errors->add(
                        'invalid_cedula',
                        sprintf(
                            __('La cédula %s no es válida. Por favor verifica el número.', 'factunexo-woocommerce'),
                            $identification_number
                        )
                    );
                }
                break;
                
            case 'RUC':
                if (!$this->validate_ruc($identification_number)) {
                    $errors->add(
                        'invalid_ruc',
                        sprintf(
                            __('El RUC %s no es válido. Por favor verifica el número.', 'factunexo-woocommerce'),
                            $identification_number
                        )
                    );
                }
                break;
                
            case 'PASAPORTE':
                // Pasaporte solo requiere longitud mínima
                if (strlen($identification_number) < 5 || strlen($identification_number) > 13) {
                    $errors->add(
                        'invalid_pasaporte',
                        __('El número de pasaporte debe tener entre 5 y 13 caracteres.', 'factunexo-woocommerce')
                    );
                }
                break;
        }
        
        return $errors;
    }
    
    /**
     * Validar cédula ecuatoriana usando el algoritmo oficial
     */
    private function validate_cedula($cedula) {
        if (strlen($cedula) !== 10) {
            return false;
        }
        
        // Algoritmo de validación de cédula ecuatoriana
        $sum = 0;
        $length = strlen($cedula);
        
        for ($i = 0; $i < $length - 1; $i++) {
            $digit = intval($cedula[$i]);
            if ($i % 2 === 0) {
                $digit *= 2;
                if ($digit > 9) {
                    $digit -= 9;
                }
            }
            $sum += $digit;
        }
        
        $check_digit = (10 - ($sum % 10)) % 10;
        return $check_digit === intval($cedula[9]);
    }
    
    /**
     * Validar RUC ecuatoriano
     */
    private function validate_ruc($ruc) {
        if (strlen($ruc) !== 13) {
            return false;
        }
        
        // Los primeros 10 dígitos deben ser una cédula válida
        $cedula_part = substr($ruc, 0, 10);
        if (!$this->validate_cedula($cedula_part)) {
            return false;
        }
        
        // Los últimos 3 dígitos deben ser 001
        $last_three = substr($ruc, 10, 3);
        return $last_three === '001';
    }
    
    /**
     * Agregar campos al checkout tradicional (shortcode)
     * Para compatibilidad con versiones antiguas de WooCommerce
     */
    public function add_fields_to_traditional_checkout() {
        echo '<div class="factunexo-billing-fields" style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 8px; padding: 20px; margin: 20px 0;">';
        echo '<h3 style="margin: 0 0 20px 0; color: #495057; font-size: 18px; font-weight: 600; border-bottom: 2px solid #0073aa; padding-bottom: 10px;">📋 ' . __('Información de Facturación', 'factunexo-woocommerce') . '</h3>';
        
        // Campo: Tipo de Identificación
        woocommerce_form_field('factunexo_identification_type', array(
            'type' => 'select',
            'label' => __('Tipo de Identificación', 'factunexo-woocommerce'),
            'required' => true,
            'class' => array('form-row-wide'),
            'clear' => true,
            'options' => array(
                '' => __('Selecciona el tipo', 'factunexo-woocommerce'),
                'CEDULA' => __('Cédula de Identidad', 'factunexo-woocommerce'),
                'RUC' => __('RUC', 'factunexo-woocommerce'),
                'PASAPORTE' => __('Pasaporte', 'factunexo-woocommerce')
            )
        ), WC()->checkout->get_value('factunexo_identification_type'));
        
        // Campo: Número de Identificación
        woocommerce_form_field('factunexo_identification_number', array(
            'type' => 'text',
            'label' => __('Número de Identificación', 'factunexo-woocommerce'),
            'required' => true,
            'class' => array('form-row-wide'),
            'clear' => true,
            'maxlength' => 13,
            'placeholder' => __('Ingresa tu número de identificación', 'factunexo-woocommerce')
        ), WC()->checkout->get_value('factunexo_identification_number'));
        
        // Campo: Nombre de la Empresa
        woocommerce_form_field('factunexo_company_name', array(
            'type' => 'text',
            'label' => __('Nombre de la Empresa', 'factunexo-woocommerce'),
            'required' => false,
            'class' => array('form-row-wide'),
            'clear' => true,
            'placeholder' => __('Solo si deseas factura a nombre de empresa', 'factunexo-woocommerce')
        ), WC()->checkout->get_value('factunexo_company_name'));
        
        // Campo: Checkbox para usar empresa
        woocommerce_form_field('factunexo_use_company', array(
            'type' => 'checkbox',
            'label' => __('Emitir factura a nombre de la empresa', 'factunexo-woocommerce'),
            'required' => false,
            'class' => array('form-row-wide'),
            'clear' => true
        ), WC()->checkout->get_value('factunexo_use_company'));
        
        echo '</div>';
    }
    
    /**
     * Validar campos en checkout tradicional
     */
    public function validate_traditional_checkout() {
        $identification_type = isset($_POST['factunexo_identification_type']) ? sanitize_text_field($_POST['factunexo_identification_type']) : '';
        $identification_number = isset($_POST['factunexo_identification_number']) ? sanitize_text_field($_POST['factunexo_identification_number']) : '';
        
        // Validar tipo de identificación
        if (empty($identification_type)) {
            wc_add_notice(__('Debes seleccionar el tipo de identificación.', 'factunexo-woocommerce'), 'error');
        }
        
        // Validar número de identificación
        if (empty($identification_number)) {
            wc_add_notice(__('Debes ingresar el número de identificación.', 'factunexo-woocommerce'), 'error');
        } else {
            // Validar según el tipo
            switch ($identification_type) {
                case 'CEDULA':
                    if (!$this->validate_cedula($identification_number)) {
                        wc_add_notice(__('La cédula ingresada no es válida.', 'factunexo-woocommerce'), 'error');
                    }
                    break;
                case 'RUC':
                    if (!$this->validate_ruc($identification_number)) {
                        wc_add_notice(__('El RUC ingresado no es válido.', 'factunexo-woocommerce'), 'error');
                    }
                    break;
            }
        }
    }
    
    /**
     * Cargar scripts y estilos del checkout
     */
    public function enqueue_checkout_scripts() {
        if (is_checkout()) {
            // CSS para Checkout Blocks
            wp_enqueue_style(
                'factunexo-checkout-blocks',
                FACTUNEXO_WC_PLUGIN_URL . 'assets/css/checkout-blocks.css',
                array(),
                FACTUNEXO_WC_VERSION
            );
            
            // Validador de Ecuador (sin dependencias)
            wp_enqueue_script(
                'ecuador-validator',
                FACTUNEXO_WC_PLUGIN_URL . 'assets/js/ecuador-validator.js',
                array(),
                FACTUNEXO_WC_VERSION,
                true
            );
            
            // Script para checkout tradicional
            wp_enqueue_script(
                'factunexo-checkout',
                FACTUNEXO_WC_PLUGIN_URL . 'assets/js/checkout.js',
                array('jquery'),
                FACTUNEXO_WC_VERSION,
                true
            );
            
            // Script de validación de checkout (depende de ecuador-validator y jquery)
            wp_enqueue_script(
                'factunexo-checkout-validation',
                FACTUNEXO_WC_PLUGIN_URL . 'assets/js/checkout-validation.js',
                array('jquery', 'ecuador-validator'),
                FACTUNEXO_WC_VERSION,
                true
            );
            
            wp_localize_script('factunexo-checkout', 'factunexo_checkout', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('factunexo_checkout_nonce'),
                'i18n' => array(
                    'country_ec' => 'EC',
                    'identification_required' => __('Por favor completa los campos de identificación.', 'factunexo-woocommerce'),
                )
            ));
            
            // Script inline para checkout tradicional
            wp_add_inline_script('factunexo-checkout', "
                jQuery(document).ready(function($) {
                    // Solo para checkout tradicional
                    if ($('.woocommerce-checkout').length > 0 && $('.wp-block-woocommerce-checkout').length === 0) {
                        console.log('Factunexo: Campos de checkout tradicional cargados');
                        
                        // Los campos siempre están visibles, solo asegurarnos de que estén disponibles
                        $('.factunexo-billing-fields').show();
                    }
                });
            ");
        }
    }
}
