/**
 * Validador de Cédulas y RUC para Ecuador
 * Basado en la normativa del SRI vigente (2025)
 * 
 * @package FactunexoWooCommerce
 */

(function() {
    'use strict';

    /**
     * Validador de documentos de Ecuador
     */
    window.EcuadorValidator = {
        
        /**
         * Valida una cédula ecuatoriana (10 dígitos)
         * Algoritmo: Módulo 10
         */
        validateCedula: function(cedula) {
            // Validar formato básico
            if (!cedula || !/^\d{10}$/.test(cedula)) {
                return {
                    valid: false,
                    message: 'La cédula debe tener exactamente 10 dígitos numéricos'
                };
            }

            try {
                // Coeficientes para módulo 10
                const coeficientes = [2, 1, 2, 1, 2, 1, 2, 1, 2];
                let suma = 0;

                // Calcular suma con coeficientes
                for (let i = 0; i < coeficientes.length; i++) {
                    let valor = parseInt(cedula.charAt(i)) * coeficientes[i];
                    suma += (valor >= 10) ? valor - 9 : valor;
                }

                // Calcular dígito verificador
                const verificador = (suma % 10 === 0) ? 0 : 10 - (suma % 10);
                const digitoVerificador = parseInt(cedula.charAt(9));

                if (verificador === digitoVerificador) {
                    return {
                        valid: true,
                        message: 'Cédula válida',
                        type: 'CEDULA'
                    };
                } else {
                    return {
                        valid: false,
                        message: 'Cédula inválida: el dígito verificador no coincide'
                    };
                }
            } catch (error) {
                return {
                    valid: false,
                    message: 'Error al validar la cédula'
                };
            }
        },

        /**
         * Valida un RUC ecuatoriano (13 dígitos)
         * Tipos:
         * - d3 < 6: Persona Natural (basado en cédula)
         * - d3 = 6: Entidad Pública
         * - d3 = 9: Empresa Privada
         */
        validateRuc: function(ruc) {
            // Validar formato básico
            if (!ruc || !/^\d{13}$/.test(ruc)) {
                return {
                    valid: false,
                    message: 'El RUC debe tener exactamente 13 dígitos numéricos'
                };
            }

            // Validar código de provincia (01-30)
            const provincia = parseInt(ruc.substring(0, 2));
            if (provincia < 1 || provincia > 30) {
                return {
                    valid: false,
                    message: 'Código de provincia inválido en el RUC'
                };
            }

            // Validar tercer dígito
            const d3 = parseInt(ruc.charAt(2));
            if (d3 === 7 || d3 === 8) {
                return {
                    valid: false,
                    message: 'Tercer dígito del RUC inválido'
                };
            }

            // Validar según tipo de RUC
            if (d3 < 6) {
                return this.validateRucNatural(ruc);
            } else if (d3 === 6) {
                return this.validateRucPublic(ruc);
            } else if (d3 === 9) {
                return this.validateRucPrivate(ruc);
            }

            return {
                valid: false,
                message: 'Tipo de RUC no reconocido'
            };
        },

        /**
         * Valida RUC de Persona Natural (basado en cédula)
         */
        validateRucNatural: function(ruc) {
            // Debe terminar en 001
            if (!ruc.endsWith('001')) {
                return {
                    valid: false,
                    message: 'El RUC de persona natural debe terminar en 001'
                };
            }

            // Validar los primeros 10 dígitos como cédula
            const cedula = ruc.substring(0, 10);
            const resultadoCedula = this.validateCedula(cedula);

            if (resultadoCedula.valid) {
                return {
                    valid: true,
                    message: 'RUC de persona natural válido',
                    type: 'RUC_NATURAL'
                };
            } else {
                return {
                    valid: false,
                    message: 'RUC de persona natural inválido: la cédula base es incorrecta'
                };
            }
        },

        /**
         * Valida RUC de Entidad Pública
         * Normativa SRI 2025: Solo valida estructura
         */
        validateRucPublic: function(ruc) {
            // Debe terminar en 0001
            if (!ruc.endsWith('0001')) {
                return {
                    valid: false,
                    message: 'El RUC de entidad pública debe terminar en 0001'
                };
            }

            return {
                valid: true,
                message: 'RUC de entidad pública válido',
                type: 'RUC_PUBLICO'
            };
        },

        /**
         * Valida RUC de Empresa Privada
         * Normativa SRI 2025: Solo valida estructura
         */
        validateRucPrivate: function(ruc) {
            // Debe terminar en 001
            if (!ruc.endsWith('001')) {
                return {
                    valid: false,
                    message: 'El RUC de empresa privada debe terminar en 001'
                };
            }

            return {
                valid: true,
                message: 'RUC de empresa privada válido',
                type: 'RUC_PRIVADO'
            };
        },

        /**
         * Valida cualquier tipo de identificación ecuatoriana
         * Detecta automáticamente si es cédula o RUC
         */
        validate: function(numero) {
            if (!numero) {
                return {
                    valid: false,
                    message: 'Debe ingresar un número de identificación'
                };
            }

            // Limpiar el número (remover espacios, guiones, etc)
            numero = numero.replace(/[^\d]/g, '');

            // Detectar tipo por longitud
            if (numero.length === 10) {
                return this.validateCedula(numero);
            } else if (numero.length === 13) {
                return this.validateRuc(numero);
            } else {
                return {
                    valid: false,
                    message: 'El número debe tener 10 dígitos (cédula) o 13 dígitos (RUC)'
                };
            }
        },

        /**
         * Obtiene información sobre el tipo de documento
         */
        getDocumentInfo: function(numero) {
            if (!numero) {
                return null;
            }

            numero = numero.replace(/[^\d]/g, '');

            const info = {
                numero: numero,
                length: numero.length,
                type: null,
                taxpayerType: null
            };

            if (numero.length === 10) {
                info.type = 'CEDULA';
                info.taxpayerType = 'NATURAL';
            } else if (numero.length === 13) {
                const d3 = parseInt(numero.charAt(2));
                info.type = 'RUC';
                
                if (d3 < 6) {
                    info.taxpayerType = 'NATURAL';
                } else if (d3 === 6 || d3 === 9) {
                    info.taxpayerType = 'JURIDICO';
                }
            }

            return info;
        }
    };

})();

