/**
 * JavaScript para el checkout de WooCommerce con Factunexo
 */

jQuery(document).ready(function($) {
    
    // Función para mostrar/ocultar campos de empresa
    function toggleCompanyFields() {
        var useCompany = $('#billing_use_company').is(':checked');
        var companyNameField = $('#billing_company_name').closest('.form-row');
        
        if (useCompany) {
            companyNameField.show();
            $('#billing_company_name').prop('required', true);
        } else {
            companyNameField.hide();
            $('#billing_company_name').prop('required', false);
        }
    }
    
    // Función para validar identificación ecuatoriana
    function validateEcuadorianId(idNumber, type) {
        if (!idNumber || !type) {
            return false;
        }
        
        // Remover espacios y guiones
        idNumber = idNumber.replace(/[^0-9]/g, '');
        
        switch (type) {
            case 'CEDULA':
                return validateCedula(idNumber);
            case 'RUC':
                return validateRuc(idNumber);
            case 'PASAPORTE':
                return idNumber.length >= 5 && idNumber.length <= 15;
            default:
                return false;
        }
    }
    
    // Validar cédula ecuatoriana
    function validateCedula(cedula) {
        if (cedula.length !== 10) {
            return false;
        }
        
        // Algoritmo de validación de cédula ecuatoriana
        var sum = 0;
        var length = cedula.length;
        
        for (var i = 0; i < length - 1; i++) {
            var digit = parseInt(cedula[i]);
            if (i % 2 === 0) {
                digit *= 2;
                if (digit > 9) {
                    digit -= 9;
                }
            }
            sum += digit;
        }
        
        var checkDigit = (10 - (sum % 10)) % 10;
        return checkDigit === parseInt(cedula[9]);
    }
    
    // Validar RUC ecuatoriano
    function validateRuc(ruc) {
        if (ruc.length !== 13) {
            return false;
        }
        
        // Los primeros 10 dígitos deben ser una cédula válida
        var cedulaPart = ruc.substring(0, 10);
        if (!validateCedula(cedulaPart)) {
            return false;
        }
        
        // Los últimos 3 dígitos deben ser 001
        var lastThree = ruc.substring(10, 13);
        return lastThree === '001';
    }
    
    // Función para mostrar errores de validación
    function showFieldError(field, message) {
        var $field = $(field);
        var $errorDiv = $field.siblings('.woocommerce-error');
        
        if ($errorDiv.length === 0) {
            $errorDiv = $('<div class="woocommerce-error">' + message + '</div>');
            $field.after($errorDiv);
        } else {
            $errorDiv.text(message);
        }
        
        $field.addClass('woocommerce-invalid');
    }
    
    // Función para limpiar errores de validación
    function clearFieldError(field) {
        var $field = $(field);
        var $errorDiv = $field.siblings('.woocommerce-error');
        
        if ($errorDiv.length > 0) {
            $errorDiv.remove();
        }
        
        $field.removeClass('woocommerce-invalid');
    }
    
    // Mostrar/ocultar campos de empresa
    $('#billing_use_company').on('change', function() {
        toggleCompanyFields();
    });
    
    // Validar campos de identificación en tiempo real
    $('#billing_identification_number').on('blur', function() {
        var $field = $(this);
        var idNumber = $field.val();
        var idType = $('#billing_identification_type').val();
        
        if (idNumber && idType) {
            if (!validateEcuadorianId(idNumber, idType)) {
                showFieldError($field, 'El número de identificación no es válido para el tipo seleccionado.');
            } else {
                clearFieldError($field);
            }
        }
    });
    
    // Validar tipo de identificación
    $('#billing_identification_type').on('change', function() {
        var $field = $(this);
        var $idNumberField = $('#billing_identification_number');
        
        if ($field.val()) {
            clearFieldError($field);
            
            // Limpiar validación del número si cambia el tipo
            if ($idNumberField.val()) {
                clearFieldError($idNumberField);
                $idNumberField.trigger('blur');
            }
        } else {
            showFieldError($field, 'Debes seleccionar el tipo de identificación.');
        }
    });
    
    // Validar nombre de empresa si se selecciona emitir a nombre de empresa
    $('#billing_company_name').on('blur', function() {
        var $field = $(this);
        var useCompany = $('#billing_use_company').is(':checked');
        
        if (useCompany && !$field.val().trim()) {
            showFieldError($field, 'Debes ingresar el nombre de la empresa si deseas emitir la factura a nombre de la empresa.');
        } else {
            clearFieldError($field);
        }
    });
    
    // Validar país y mostrar/ocultar campos de facturación
    $('#billing_country').on('change', function() {
        var country = $(this).val();
        var $factunexoFields = $('.factunexo-field').closest('.form-row');
        
        if (country === 'EC') {
            $factunexoFields.show();
            
            // Hacer obligatorios los campos de identificación
            $('#billing_identification_type').prop('required', true);
            $('#billing_identification_number').prop('required', true);
            
            // Agregar clase de validación
            $('#billing_identification_type').closest('.form-row').addClass('validate-required');
            $('#billing_identification_number').closest('.form-row').addClass('validate-required');
        } else {
            $factunexoFields.hide();
            
            // Quitar obligatoriedad de los campos de identificación
            $('#billing_identification_type').prop('required', false);
            $('#billing_identification_number').prop('required', false);
            
            // Quitar clase de validación
            $('#billing_identification_type').closest('.form-row').removeClass('validate-required');
            $('#billing_identification_number').closest('.form-row').removeClass('validate-required');
            
            // Limpiar errores
            clearFieldError('#billing_identification_type');
            clearFieldError('#billing_identification_number');
            clearFieldError('#billing_company_name');
        }
    });
    
    // Validar formulario antes de enviar
    $('form.checkout').on('checkout_place_order', function() {
        var country = $('#billing_country').val();
        
        if (country === 'EC') {
            var isValid = true;
            
            // Validar tipo de identificación
            var idType = $('#billing_identification_type').val();
            if (!idType) {
                showFieldError('#billing_identification_type', 'Debes seleccionar el tipo de identificación.');
                isValid = false;
            }
            
            // Validar número de identificación
            var idNumber = $('#billing_identification_number').val();
            if (!idNumber) {
                showFieldError('#billing_identification_number', 'Debes ingresar el número de identificación.');
                isValid = false;
            } else if (!validateEcuadorianId(idNumber, idType)) {
                showFieldError('#billing_identification_number', 'El número de identificación no es válido.');
                isValid = false;
            }
            
            // Validar nombre de empresa si se selecciona emitir a nombre de empresa
            var useCompany = $('#billing_use_company').is(':checked');
            var companyName = $('#billing_company_name').val();
            if (useCompany && !companyName.trim()) {
                showFieldError('#billing_company_name', 'Debes ingresar el nombre de la empresa.');
                isValid = false;
            }
            
            if (!isValid) {
                // Scroll al primer error
                $('.woocommerce-error:first').get(0).scrollIntoView({ behavior: 'smooth', block: 'center' });
                return false;
            }
        }
        
        return true;
    });
    
    // Formatear número de identificación automáticamente
    $('#billing_identification_number').on('input', function() {
        var $field = $(this);
        var value = $field.val();
        
        // Solo permitir números
        value = value.replace(/[^0-9]/g, '');
        
        // Formatear según el tipo
        var idType = $('#billing_identification_type').val();
        if (idType === 'CEDULA' && value.length > 10) {
            value = value.substring(0, 10);
        } else if (idType === 'RUC' && value.length > 13) {
            value = value.substring(0, 13);
        }
        
        $field.val(value);
    });
    
    // Auto-completar tipo de identificación basado en la longitud
    $('#billing_identification_number').on('blur', function() {
        var $field = $(this);
        var value = $field.val();
        var $idTypeField = $('#billing_identification_type');
        
        if (value && !$idTypeField.val()) {
            if (value.length === 10) {
                $idTypeField.val('CEDULA');
            } else if (value.length === 13) {
                $idTypeField.val('RUC');
            }
        }
    });
    
    // Mostrar campos de facturación si el país es Ecuador al cargar la página
    $(document).ready(function() {
        var country = $('#billing_country').val();
        var $factunexoFields = $('.factunexo-field').closest('.form-row');
        
        if (country === 'EC') {
            $factunexoFields.show();
            
            // Hacer obligatorios los campos de identificación
            $('#billing_identification_type').prop('required', true);
            $('#billing_identification_number').prop('required', true);
            
            // Agregar clase de validación
            $('#billing_identification_type').closest('.form-row').addClass('validate-required');
            $('#billing_identification_number').closest('.form-row').addClass('validate-required');
        } else {
            $factunexoFields.hide();
        }
    });
    
    // Tooltips para campos de ayuda
    $('.factunexo-billing-fields .description').each(function() {
        var $description = $(this);
        var $field = $description.prev('input, select');
        
        if ($field.length) {
            $field.attr('title', $description.text());
        }
    });
    
    // Mostrar indicador de carga durante la validación
    $('form.checkout').on('checkout_place_order', function() {
        var $submitButton = $('button#place_order');
        var originalText = $submitButton.text();
        
        $submitButton.text('Validando...').prop('disabled', true);
        
        // Re-habilitar después de un tiempo si hay error
        setTimeout(function() {
            $submitButton.text(originalText).prop('disabled', false);
        }, 5000);
    });
});
