/**
 * Validación de campos de checkout para Factunexo
 * Integración con WooCommerce Checkout Blocks
 * 
 * @package FactunexoWooCommerce
 */

jQuery(document).ready(function($) {
    'use strict';

    console.log('Factunexo: Inicializando validación de campos');

    /**
     * Validar campo de identificación en tiempo real
     */
    function setupIdentificationValidation() {
        // Selector para el campo de número de identificación
        const identificationSelector = 'input[name="billing-factunexo/identification-number"]';
        
        // Función para validar el campo
        function validateField($field) {
            const numero = $field.val();
            const $container = $field.closest('.wc-block-components-text-input');
            const $errorContainer = $container.find('.factunexo-validation-error');
            const $successIcon = $container.find('.factunexo-validation-success');
            
            // Remover feedback anterior
            $field.removeClass('has-error has-success');
            $errorContainer.remove();
            $successIcon.remove();

            if (!numero) {
                return true; // No validar si está vacío (WooCommerce maneja campos requeridos)
            }

            // Validar usando EcuadorValidator
            if (typeof window.EcuadorValidator !== 'undefined') {
                const resultado = window.EcuadorValidator.validate(numero);
                
                if (!resultado.valid) {
                    // Mostrar error
                    $field.addClass('has-error');
                    const $error = $('<div class="factunexo-validation-error"></div>');
                    $error.html('<span style="color: #dc3232;">❌</span> ' + resultado.message);
                    $container.append($error);
                    
                    console.log('Factunexo: Validación fallida -', resultado.message);
                    return false;
                } else {
                    // Mostrar éxito
                    $field.addClass('has-success');
                    const $success = $('<div class="factunexo-validation-success"></div>');
                    $success.html('<span style="color: #46b450;">✓</span> ' + resultado.message);
                    $container.append($success);
                    
                    console.log('Factunexo: Validación exitosa -', resultado.type);
                    return true;
                }
            }
            
            return true;
        }
        
        // Validar en blur, change y input (tiempo real)
        $(document).on('blur change input', identificationSelector, function() {
            const $field = $(this);
            // Debounce para input (no validar en cada tecla, solo después de 500ms de inactividad)
            clearTimeout($field.data('validateTimeout'));
            const timeout = setTimeout(function() {
                validateField($field);
            }, 500);
            $field.data('validateTimeout', timeout);
        });
        
        // Validar inmediatamente en blur y change
        $(document).on('blur change', identificationSelector, function() {
            const $field = $(this);
            clearTimeout($field.data('validateTimeout'));
            validateField($field);
        });
    }

    /**
     * Validar en el submit del checkout
     */
    function setupCheckoutValidation() {
        // WooCommerce Blocks - Validación del campo antes del submit
        $(document).on('click', '.wc-block-components-checkout-place-order-button', function(e) {
            const $button = $(this);
            const $identificacionField = $('input[name="billing-factunexo/identification-number"]');
            
            if ($identificacionField.length && $identificacionField.val()) {
                const numero = $identificacionField.val();
                
                if (typeof window.EcuadorValidator !== 'undefined') {
                    const resultado = window.EcuadorValidator.validate(numero);
                    
                    if (!resultado.valid) {
                        e.preventDefault();
                        e.stopImmediatePropagation();
                        
                        // Mostrar notificación de error prominente
                        showErrorNotification(resultado.message);
                        
                        // Scroll al campo con error
                        $('html, body').animate({
                            scrollTop: $identificacionField.offset().top - 100
                        }, 500);
                        
                        // Agregar clase de error al campo
                        $identificacionField.addClass('has-error');
                        
                        console.log('Factunexo: Submit bloqueado - Validación fallida');
                        return false;
                    }
                }
            }
        });
        
        // WooCommerce Tradicional
        $('form.checkout').on('checkout_place_order', function() {
            const identificacion = $('input[name="billing_identification_number"]').val() || 
                                 $('input[name="billing-factunexo/identification-number"]').val();
            
            if (identificacion && typeof window.EcuadorValidator !== 'undefined') {
                const resultado = window.EcuadorValidator.validate(identificacion);
                
                if (!resultado.valid) {
                    // Mostrar notificación de error prominente
                    showErrorNotification(resultado.message);
                    
                    // Scroll al campo con error
                    const $field = $('input[name="billing_identification_number"]').length ? 
                                  $('input[name="billing_identification_number"]') :
                                  $('input[name="billing-factunexo/identification-number"]');
                    
                    $('html, body').animate({
                        scrollTop: $field.offset().top - 100
                    }, 500);
                    
                    console.log('Factunexo: Validación tradicional fallida');
                    return false; // Prevenir submit
                }
            }
            
            return true; // Permitir submit
        });
    }
    
    /**
     * Mostrar notificación de error prominente
     */
    function showErrorNotification(message) {
        // Remover notificaciones anteriores
        $('.factunexo-error-notification').remove();
        
        // Crear notificación
        const $notification = $('<div class="factunexo-error-notification"></div>');
        $notification.html(
            '<div style="background: #dc3232; color: white; padding: 15px 20px; border-radius: 4px; margin: 15px 0; font-weight: 500; box-shadow: 0 2px 8px rgba(0,0,0,0.2);">' +
            '<span style="font-size: 18px; margin-right: 10px;">⚠️</span>' +
            '<strong>Error de validación:</strong> ' + message +
            '</div>'
        );
        
        // Insertar en WooCommerce Blocks
        const $blocksContainer = $('.wc-block-checkout__main');
        if ($blocksContainer.length) {
            $blocksContainer.prepend($notification);
        } else {
            // Insertar en checkout tradicional
            $('form.checkout').prepend($notification);
        }
        
        // Auto-remover después de 8 segundos
        setTimeout(function() {
            $notification.fadeOut(function() {
                $(this).remove();
            });
        }, 8000);
    }

    /**
     * Agregar indicador visual de validación
     */
    function setupVisualFeedback() {
        const style = $('<style></style>');
        style.text(`
            .factunexo-validation-error {
                color: #dc3232;
                font-size: 13px;
                margin-top: 8px;
                font-weight: 500;
                line-height: 1.4;
            }
            .factunexo-validation-success {
                color: #46b450;
                font-size: 13px;
                margin-top: 8px;
                font-weight: 500;
                line-height: 1.4;
            }
            input.has-error {
                border-color: #dc3232 !important;
                box-shadow: 0 0 0 1px #dc3232 !important;
            }
            input.has-success {
                border-color: #46b450 !important;
                box-shadow: 0 0 0 1px #46b450 !important;
            }
            .factunexo-error-notification {
                animation: slideDown 0.3s ease-out;
            }
            @keyframes slideDown {
                from {
                    opacity: 0;
                    transform: translateY(-20px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
        `);
        $('head').append(style);
    }

    /**
     * Inicializar validaciones
     */
    function init() {
        // Esperar a que EcuadorValidator esté disponible
        if (typeof window.EcuadorValidator === 'undefined') {
            console.warn('Factunexo: EcuadorValidator no está disponible');
            return;
        }

        setupVisualFeedback();
        setupIdentificationValidation();
        setupCheckoutValidation();
        
        console.log('Factunexo: Validación de campos inicializada correctamente');
    }

    // Inicializar cuando el DOM esté listo
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
});

