/**
 * JavaScript para la administración del plugin Factunexo WooCommerce
 */

jQuery(document).ready(function($) {
    
    // Función para mostrar notificaciones
    function showNotification(message, type) {
        var noticeClass = type === 'error' ? 'notice-error' : 'notice-success';
        var notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
        
        $('.wrap h1').after(notice);
        
        // Auto-dismiss después de 5 segundos
        setTimeout(function() {
            notice.fadeOut();
        }, 5000);
    }
    
    // Función para validar campos requeridos
    function validateRequiredFields() {
        var isValid = true;
        var requiredFields = ['#factunexo_wc_api_key', '#factunexo_wc_api_secret'];
        
        requiredFields.forEach(function(field) {
            var $field = $(field);
            if (!$field.val().trim()) {
                $field.addClass('error');
                isValid = false;
            } else {
                $field.removeClass('error');
            }
        });
        
        return isValid;
    }
    
    // Validar campos antes de guardar
    $('#submit').on('click', function(e) {
        if (!validateRequiredFields()) {
            e.preventDefault();
            showNotification('Por favor, completa todos los campos requeridos.', 'error');
            return false;
        }
    });
    
    // Probar conexión con la API
    $('#test_connection').on('click', function() {
        var button = $(this);
        var resultDiv = $('#connection_result');
        
        if (!validateRequiredFields()) {
            showNotification('Por favor, completa los campos de API Key y Secret antes de probar la conexión.', 'error');
            return;
        }
        
        button.prop('disabled', true).text('Probando conexión...');
        resultDiv.hide();
        
        $.ajax({
            url: factunexo_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'factunexo_test_connection',
                nonce: factunexo_ajax.nonce
            },
            success: function(response) {
                console.log('Connection Test Response:', response); // Debug
                
                if (response.success) {
                    var message = response.data.message;
                    
                    // Agregar información adicional si está disponible
                    if (response.data.api_url) {
                        message += '<br><small>URL de API: ' + response.data.api_url + '</small>';
                    }
                    
                    showNotification(message, 'success');
                    
                    // Mostrar información sobre puntos de venta disponibles
                    if (response.data.pos_count > 0) {
                        showNotification('Conexión exitosa. Tienes ' + response.data.pos_count + ' puntos de venta disponibles. Haz clic en "Cargar puntos de venta" para verlos.', 'success');
                    } else {
                        showNotification('Conexión exitosa, pero no se encontraron puntos de venta en tu cuenta.', 'success');
                    }
                } else {
                    var errorMessage = 'Error de conexión: ' + response.data;
                    
                    // Agregar información de depuración si está disponible
                    if (response.data && typeof response.data === 'object') {
                        if (response.data.error_type) {
                            errorMessage += '<br><small>Tipo de error: ' + response.data.error_type + '</small>';
                        }
                        if (response.data.error_data && response.data.error_data.status) {
                            errorMessage += '<br><small>Código HTTP: ' + response.data.error_data.status + '</small>';
                        }
                    }
                    
                    showNotification(errorMessage, 'error');
                }
            },
            error: function() {
                showNotification('Error de conexión con el servidor.', 'error');
            },
            complete: function() {
                button.prop('disabled', false).text('Probar conexión');
            }
        });
    });
    
    // Cargar catálogo de puntos de venta
    function loadPosCatalog() {
        var button = $('#load_pos_catalog');
        button.prop('disabled', true).text('Cargando...');
        
        $.ajax({
            url: factunexo_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'factunexo_get_pos_catalog',
                nonce: factunexo_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    var select = $('#factunexo_wc_pos_id');
                    var currentValue = select.val(); // Guardar valor actual
                    
                    select.empty().append('<option value="">Selecciona un punto de venta</option>');
                    
                    if (response.data && response.data.length > 0) {
                        $.each(response.data, function(index, pos) {
                            var selected = (pos.id == currentValue) ? 'selected' : '';
                            select.append('<option value="' + pos.id + '" ' + selected + '>' + pos.name + '</option>');
                        });
                        
                        button.text('Puntos de venta cargados (' + response.data.length + ')');
                        showNotification('Se cargaron ' + response.data.length + ' puntos de venta exitosamente.', 'success');
                        
                        // Mostrar estado
                        $('#pos_status').show();
                        $('#pos_count').text(response.data.length);
                    } else {
                        button.text('No hay puntos de venta disponibles');
                        showNotification('No se encontraron puntos de venta en tu cuenta.', 'error');
                        
                        // Mostrar estado
                        $('#pos_status').show();
                        $('#pos_count').text('0');
                    }
                } else {
                    button.text('Error al cargar');
                    showNotification('Error al cargar puntos de venta: ' + response.data, 'error');
                }
            },
            error: function() {
                button.text('Error de conexión');
                showNotification('Error de conexión al cargar puntos de venta.', 'error');
            },
            complete: function() {
                button.prop('disabled', false);
            }
        });
    }
    
    // Cargar puntos de venta al hacer clic
    $('#load_pos_catalog').on('click', function() {
        loadPosCatalog();
    });
    
    // Validar campos en tiempo real
    $('#factunexo_wc_api_key, #factunexo_wc_api_secret').on('blur', function() {
        var $field = $(this);
        if (!$field.val().trim()) {
            $field.addClass('error');
        } else {
            $field.removeClass('error');
        }
    });
    
    // Mostrar/ocultar secret de la API
    $('#factunexo_wc_api_secret').on('focus', function() {
        $(this).attr('type', 'text');
    }).on('blur', function() {
        $(this).attr('type', 'password');
    });
    
    // Confirmar antes de cambiar configuración crítica
    $('#factunexo_wc_invoice_trigger').on('change', function() {
        var newValue = $(this).val();
        var currentValue = $(this).find('option:selected').text();
        
        if (confirm('¿Estás seguro de que quieres cambiar el momento de emisión de facturas a "' + currentValue + '"?\n\nEsto afectará a todos los pedidos futuros.')) {
            return true;
        } else {
            // Revertir al valor anterior
            $(this).val($(this).data('previous-value') || 'processing');
            return false;
        }
    });
    
    // Guardar valor anterior del trigger
    $('#factunexo_wc_invoice_trigger').on('focus', function() {
        $(this).data('previous-value', $(this).val());
    });
    
    // Mostrar JSON de debug de orden
    $('#show_json_debug').on('click', function() {
        var button = $(this);
        var orderId = $('#json_debug_order_id').val();
        var infoDiv = $('#json_debug_info');
        
        if (!orderId) {
            infoDiv.html('<div class="notice notice-error"><p>Por favor, ingresa un ID de pedido.</p></div>').show();
            return;
        }
        
        button.prop('disabled', true).text('Cargando...');
        infoDiv.hide();
        
        $.ajax({
            url: factunexo_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'factunexo_show_json_debug',
                nonce: factunexo_ajax.nonce,
                order_id: orderId
            },
            success: function(response) {
                if (response.success) {
                    var jsonHtml = '<div class="notice notice-success" style="max-width: 100%;">';
                    jsonHtml += '<p><strong>JSON enviado para el pedido #' + response.data.order_id + ':</strong></p>';
                    jsonHtml += '<pre style="background: #f5f5f5; padding: 15px; border-radius: 4px; overflow-x: auto; white-space: pre-wrap;">';
                    jsonHtml += response.data.json;
                    jsonHtml += '</pre>';
                    jsonHtml += '<p><small><strong>Nota:</strong> Este JSON se guardó cuando se intentó crear la factura.</small></p>';
                    jsonHtml += '</div>';
                    
                    infoDiv.html(jsonHtml).show();
                } else {
                    infoDiv.html('<div class="notice notice-error"><p>Error: ' + response.data + '</p></div>').show();
                }
            },
            error: function() {
                infoDiv.html('<div class="notice notice-error"><p>Error de conexión al obtener el JSON.</p></div>').show();
            },
            complete: function() {
                button.prop('disabled', false).text('Mostrar JSON');
            }
        });
    });
});
