<?php
/**
 * Clase de administración para el panel de WordPress
 *
 * @package FactunexoWooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class Factunexo_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'init_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_factunexo_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_factunexo_get_pos_catalog', array($this, 'ajax_get_pos_catalog'));
        add_action('wp_ajax_factunexo_check_error_logs', array($this, 'ajax_check_error_logs'));
        add_action('wp_ajax_factunexo_test_basic_connectivity', array($this, 'ajax_test_basic_connectivity'));
        add_action('wp_ajax_factunexo_debug_order', array($this, 'ajax_debug_order'));
        add_action('wp_ajax_factunexo_show_json_debug', array($this, 'ajax_show_json_debug'));
    }
    
    /**
     * Agregar menú de administración
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            'Factunexo',
            'Factunexo',
            'manage_woocommerce',
            'factunexo-settings',
            array($this, 'admin_page')
        );
    }
    
    /**
     * Inicializar configuraciones
     */
    public function init_settings() {
        register_setting('factunexo_settings', 'factunexo_wc_api_key');
        register_setting('factunexo_settings', 'factunexo_wc_api_secret');
        register_setting('factunexo_settings', 'factunexo_wc_api_url');
        register_setting('factunexo_settings', 'factunexo_wc_invoice_trigger');
        register_setting('factunexo_settings', 'factunexo_wc_pos_id');
        register_setting('factunexo_settings', 'factunexo_wc_auto_create_items');
        register_setting('factunexo_settings', 'factunexo_wc_auto_create_customers');
        register_setting('factunexo_settings', 'factunexo_wc_merchant_id');
    }
    
    /**
     * Cargar scripts de administración
     */
    public function enqueue_admin_scripts($hook) {
        if ('woocommerce_page_factunexo-settings' !== $hook) {
            return;
        }
        
        wp_enqueue_script(
            'factunexo-admin',
            FACTUNEXO_WC_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            FACTUNEXO_WC_VERSION,
            true
        );
        
        wp_localize_script('factunexo-admin', 'factunexo_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('factunexo_nonce'),
            'strings' => array(
                'testing_connection' => 'Probando conexión...',
                'connection_success' => 'Conexión exitosa',
                'connection_error' => 'Error de conexión',
                'loading_pos' => 'Cargando puntos de venta...'
            )
        ));
    }
    
    /**
     * Página de administración
     */
    public function admin_page() {
        $api = new Factunexo_API();
        $connection_test = null;
        
        if (isset($_POST['test_connection'])) {
            $connection_test = $api->test_connection();
        }
        
        ?>
        <div class="wrap">
            <h1>Configuración de Factunexo</h1>
            <p>Configura la integración con Factunexo para facturación electrónica automática.</p>
            
            <form method="post" action="options.php">
                <?php settings_fields('factunexo_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="factunexo_wc_api_key">API Key</label>
                        </th>
                        <td>
                            <input type="text" id="factunexo_wc_api_key" name="factunexo_wc_api_key" 
                                   value="<?php echo esc_attr(get_option('factunexo_wc_api_key')); ?>" 
                                   class="regular-text" required />
                            <p class="description">Ingresa tu API Key de Factunexo</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="factunexo_wc_api_secret">API Secret</label>
                        </th>
                        <td>
                            <input type="password" id="factunexo_wc_api_secret" name="factunexo_wc_api_secret" 
                                   value="<?php echo esc_attr(get_option('factunexo_wc_api_secret')); ?>" 
                                   class="regular-text" required />
                            <p class="description">Ingresa tu API Secret de Factunexo</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="factunexo_wc_api_url">URL de la API</label>
                        </th>
                        <td>
                            <input type="url" id="factunexo_wc_api_url" name="factunexo_wc_api_url" 
                                   value="<?php echo esc_attr(get_option('factunexo_wc_api_url', 'https://api.factunexo.com')); ?>" 
                                   class="regular-text" />
                            <p class="description">URL base de la API de Factunexo</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="factunexo_wc_merchant_id">Merchant ID</label>
                        </th>
                        <td>
                            <input type="text" id="factunexo_wc_merchant_id" name="factunexo_wc_merchant_id" 
                                   value="<?php echo esc_attr(get_option('factunexo_wc_merchant_id')); ?>" 
                                   class="regular-text" />
                            <p class="description">ID del merchant para el contexto de tenant (opcional)</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="factunexo_wc_invoice_trigger">Momento de emisión</label>
                        </th>
                        <td>
                            <select id="factunexo_wc_invoice_trigger" name="factunexo_wc_invoice_trigger">
                                <option value="processing" <?php selected(get_option('factunexo_wc_invoice_trigger'), 'processing'); ?>>
                                    Cuando el pedido pase a "Procesando"
                                </option>
                                <option value="completed" <?php selected(get_option('factunexo_wc_invoice_trigger'), 'completed'); ?>>
                                    Cuando el pedido pase a "Completado"
                                </option>
                            </select>
                            <p class="description">Selecciona cuándo se debe emitir la factura automáticamente</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="factunexo_wc_pos_id">Punto de Venta</label>
                        </th>
                        <td>
                            <select id="factunexo_wc_pos_id" name="factunexo_wc_pos_id">
                                <option value="">Selecciona un punto de venta</option>
                            </select>
                            <p class="description">Punto de venta desde donde se emitirán las facturas</p>
                            <button type="button" id="load_pos_catalog" class="button">Cargar puntos de venta</button>
                            <div id="pos_status" style="margin-top: 10px; padding: 8px; background: #f9f9f9; border-radius: 4px; display: none;">
                                <small>Estado: <span id="pos_count">0</span> puntos de venta disponibles</small>
                            </div>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Opciones automáticas</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="checkbox" name="factunexo_wc_auto_create_items" value="yes" 
                                           <?php checked(get_option('factunexo_wc_auto_create_items'), 'yes'); ?> />
                                    Crear productos automáticamente si no existen
                                </label>
                                <br>
                                <label>
                                    <input type="checkbox" name="factunexo_wc_auto_create_customers" value="yes" 
                                           <?php checked(get_option('factunexo_wc_auto_create_customers'), 'yes'); ?> />
                                    Crear clientes automáticamente si no existen
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button button-primary" value="Guardar cambios">
                    <button type="button" id="test_connection" class="button">Probar conexión</button>
                </p>
            </form>
            
            <?php if ($connection_test): ?>
                <div class="notice notice-<?php echo $connection_test['success'] ? 'success' : 'error'; ?> is-dismissible">
                    <p><?php echo esc_html($connection_test['message']); ?></p>
                </div>
            <?php endif; ?>
            
            <div id="connection_result" style="display: none;"></div>
            
            <!-- Panel de depuración -->
            <div class="postbox" style="margin-top: 20px;">
                <h3 class="hndle"><span>Información de Depuración</span></h3>
                <div class="inside">
                    <p>Esta información te ayudará a diagnosticar problemas de conexión con la API:</p>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Estado de configuración:</th>
                            <td>
                                <?php 
                                $api = new Factunexo_API();
                                if ($api->is_configured()) {
                                    echo '<span style="color: green;">✓ Configurado</span>';
                                } else {
                                    echo '<span style="color: red;">✗ No configurado</span>';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">API Key:</th>
                            <td>
                                <?php 
                                $api_key = get_option('factunexo_wc_api_key');
                                if ($api_key) {
                                    echo '<span style="color: green;">✓ Configurado</span> (' . substr($api_key, 0, 8) . '...' . substr($api_key, -4) . ')';
                                } else {
                                    echo '<span style="color: red;">✗ No configurado</span>';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">API Secret:</th>
                            <td>
                                <?php 
                                $api_secret = get_option('factunexo_wc_api_secret');
                                if ($api_secret) {
                                    echo '<span style="color: green;">✓ Configurado</span> (' . substr($api_secret, 0, 8) . '...' . substr($api_secret, -4) . ')';
                                } else {
                                    echo '<span style="color: red;">✗ No configurado</span>';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">URL de API:</th>
                            <td>
                                <?php echo esc_html(get_option('factunexo_wc_api_url', 'https://api.factunexo.com')); ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Pruebas de conectividad:</th>
                            <td>
                                <button type="button" id="test_basic_connectivity" class="button">Probar conectividad básica</button>
                                <button type="button" id="check_error_logs" class="button">Verificar logs recientes</button>
                                <div id="connectivity_test_info" style="margin-top: 10px; display: none;"></div>
                                <div id="error_logs_info" style="margin-top: 10px; display: none;"></div>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Depuración de pedidos:</th>
                            <td>
                                <input type="number" id="debug_order_id" placeholder="ID del pedido" style="width: 150px;" />
                                <button type="button" id="debug_order" class="button">Depurar pedido</button>
                                <div id="debug_order_info" style="margin-top: 10px; display: none;"></div>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- Panel de JSON Debug -->
            <div class="postbox" style="margin-top: 20px;">
                <h3 class="hndle"><span>JSON Debug de Órdenes</span></h3>
                <div class="inside">
                    <p>Esta sección muestra el JSON que se envía a la API para las órdenes:</p>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Ver JSON de orden:</th>
                            <td>
                                <input type="number" id="json_debug_order_id" placeholder="ID del pedido" style="width: 150px;" />
                                <button type="button" id="show_json_debug" class="button">Mostrar JSON</button>
                                <div id="json_debug_info" style="margin-top: 10px; display: none;"></div>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Función para cargar puntos de venta
            function loadPosCatalog(showLoading = true) {
                var button = $('#load_pos_catalog');
                var select = $('#factunexo_wc_pos_id');
                
                if (showLoading) {
                    button.prop('disabled', true).text('Cargando...');
                    $('#pos_status').html('<div class="notice notice-info"><p>🔄 Cargando puntos de venta...</p></div>');
                }
                
                $.ajax({
                    url: factunexo_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'factunexo_get_pos_catalog',
                        nonce: factunexo_ajax.nonce
                    },
                    success: function(response) {
                        console.log('POS Response:', response); // Debug
                        
                        if (response.success && response.data && Array.isArray(response.data) && response.data.length > 0) {
                            select.empty().append('<option value="">Selecciona un punto de venta</option>');
                            
                            $.each(response.data, function(index, pos) {
                                console.log('POS Item:', pos); // Debug
                                if (pos && pos.value && pos.label) {
                                    select.append('<option value="' + pos.value + '">' + pos.label + '</option>');
                                } else {
                                    console.warn('POS item inválido:', pos);
                                }
                            });
                            
                            button.text('Recargar puntos de venta (' + response.data.length + ')');
                            
                            // Mostrar información adicional
                            $('#pos_status').html('<div class="notice notice-success"><p>✅ Se cargaron ' + response.data.length + ' puntos de venta correctamente.</p></div>');
                            
                            // Seleccionar el primer POS si solo hay uno
                            if (response.data.length === 1) {
                                select.val(response.data[0].value).trigger('change');
                            }
                            
                        } else if (response.success && (!response.data || !Array.isArray(response.data) || response.data.length === 0)) {
                            button.text('Cargar puntos de venta');
                            $('#pos_status').html('<div class="notice notice-warning"><p>⚠️ No se encontraron puntos de venta disponibles.</p></div>');
                        } else {
                            button.text('Cargar puntos de venta');
                            $('#pos_status').html('<div class="notice notice-error"><p>❌ Error al cargar puntos de venta: ' + (response.data || 'Respuesta inválida') + '</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX Error:', {xhr: xhr, status: status, error: error}); // Debug
                        button.text('Cargar puntos de venta');
                        $('#pos_status').html('<div class="notice notice-error"><p>❌ Error de conexión: ' + error + '</p></div>');
                    },
                    complete: function() {
                        if (showLoading) {
                            button.prop('disabled', false);
                        }
                    }
                });
            }
            
            // Cargar puntos de venta automáticamente al abrir la página
            function autoLoadPosCatalog() {
                // Solo cargar si hay API key configurada
                var apiKey = $('#factunexo_wc_api_key').val();
                var apiSecret = $('#factunexo_wc_api_secret').val();
                
                if (apiKey && apiSecret) {
                    console.log('API configurada detectada, cargando puntos de venta automáticamente...');
                    loadPosCatalog(true);
                } else {
                    console.log('API no configurada, esperando configuración manual...');
                    $('#pos_status').html('<div class="notice notice-info"><p>ℹ️ Configura tu API Key y Secret para cargar puntos de venta automáticamente.</p></div>');
                }
            }
            
            // Cargar puntos de venta al hacer clic manual
            $('#load_pos_catalog').on('click', function() {
                loadPosCatalog(true);
            });
            
            // Cargar automáticamente cuando se configuran las credenciales
            $('#factunexo_wc_api_key, #factunexo_wc_api_secret').on('change', function() {
                var apiKey = $('#factunexo_wc_api_key').val();
                var apiSecret = $('#factunexo_wc_api_secret').val();
                
                if (apiKey && apiSecret) {
                    // Esperar un poco para que se guarden las credenciales
                    setTimeout(function() {
                        console.log('Credenciales configuradas, cargando puntos de venta...');
                        loadPosCatalog(true);
                    }, 1000);
                }
            });
            
            // Probar conexión
            $('#test_connection').on('click', function() {
                var button = $(this);
                var resultDiv = $('#connection_result');
                
                button.prop('disabled', true).text('Probando...');
                resultDiv.hide();
                
                $.ajax({
                    url: factunexo_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'factunexo_test_connection',
                        nonce: factunexo_ajax.nonce
                    },
                    success: function(response) {
                        console.log('Connection Test Response:', response); // Debug
                        
                        if (response.success) {
                            var message = response.data.message;
                            if (response.data.pos_count !== undefined) {
                                message += '<br><strong>Puntos de venta disponibles:</strong> ' + response.data.pos_count;
                                if (response.data.pos_count > 0) {
                                    message += '<br><em>💡 Los puntos de venta se cargarán automáticamente.</em>';
                                }
                            }
                            
                            resultDiv.removeClass('notice-error').addClass('notice notice-success')
                                   .html('<p>' + message + '</p>').show();
                        } else {
                            resultDiv.removeClass('notice-success').addClass('notice notice-error')
                                   .html('<p>Error: ' + response.data + '</p>').show();
                        }
                    },
                    error: function() {
                        resultDiv.removeClass('notice-success').addClass('notice notice-error')
                               .html('<p>Error de conexión</p>').show();
                    },
                    complete: function() {
                        button.prop('disabled', false).text('Probar conexión');
                    }
                });
            });
            
            // Probar conectividad básica
            $('#test_basic_connectivity').on('click', function() {
                var button = $(this);
                var infoDiv = $('#connectivity_test_info');
                
                button.prop('disabled', true).text('Probando...');
                infoDiv.hide();
                
                $.ajax({
                    url: factunexo_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'factunexo_test_basic_connectivity',
                        nonce: factunexo_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            var message = '✅ Conectividad básica exitosa<br>';
                            message += '<strong>URL:</strong> ' + response.data.url + '<br>';
                            message += '<strong>Código de respuesta:</strong> ' + response.data.response_code + '<br>';
                            
                            if (response.data.response_body) {
                                message += '<strong>Respuesta del servidor:</strong><br>';
                                message += '<pre style="background: #f9f9f9; padding: 10px; border-radius: 4px; font-size: 12px;">' + response.data.response_body + '</pre>';
                            }
                            
                            infoDiv.html('<div class="notice notice-success"><p>' + message + '</p></div>').show();
                        } else {
                            var message = '❌ Error de conectividad: ' + response.data;
                            infoDiv.html('<div class="notice notice-error"><p>' + message + '</p></div>').show();
                        }
                    },
                    error: function() {
                        infoDiv.html('<div class="notice notice-error"><p>Error de conexión al probar conectividad.</p></div>').show();
                    },
                    complete: function() {
                        button.prop('disabled', false).text('Probar conectividad básica');
                    }
                });
            });
            
            // Verificar logs de error
            $('#check_error_logs').on('click', function() {
                var button = $(this);
                var infoDiv = $('#error_logs_info');
                
                button.prop('disabled', true).text('Verificando...');
                infoDiv.hide();
                
                $.ajax({
                    url: factunexo_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'factunexo_check_error_logs',
                        nonce: factunexo_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            var logs = response.data.logs;
                            if (logs && logs.length > 0) {
                                var logHtml = '<div class="notice notice-warning"><p><strong>Logs recientes de Factunexo:</strong></p><ul>';
                                logs.forEach(function(log) {
                                    logHtml += '<li>' + log + '</li>';
                                });
                                logHtml += '</ul></div>';
                                infoDiv.html(logHtml).show();
                            } else {
                                infoDiv.html('<div class="notice notice-success"><p>✓ No se encontraron errores recientes de Factunexo en los logs.</p></div>').show();
                            }
                        } else {
                            infoDiv.html('<div class="notice notice-error"><p>Error al verificar logs: ' + response.data + '</p></div>').show();
                        }
                    },
                    error: function() {
                        infoDiv.html('<div class="notice notice-error"><p>Error de conexión al verificar logs.</p></div>').show();
                    },
                    complete: function() {
                        button.prop('disabled', false).text('Verificar logs recientes');
                    }
                });
            });
            
            // Depurar pedido
            $('#debug_order').on('click', function() {
                var button = $(this);
                var orderId = $('#debug_order_id').val();
                var infoDiv = $('#debug_order_info');
                
                if (!orderId) {
                    infoDiv.html('<div class="notice notice-error"><p>Por favor, ingresa un ID de pedido.</p></div>').show();
                    return;
                }
                
                button.prop('disabled', true).text('Depurando...');
                infoDiv.hide();
                
                $.ajax({
                    url: factunexo_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'factunexo_debug_order',
                        order_id: orderId,
                        nonce: factunexo_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            var data = response.data;
                            var debugHtml = '<div class="notice notice-info"><p><strong>Información del pedido #' + data.order_id + ':</strong></p>';
                            debugHtml += '<ul>';
                            debugHtml += '<li><strong>País de facturación:</strong> ' + (data.billing_country || 'No configurado') + '</li>';
                            debugHtml += '<li><strong>Tipo de identificación:</strong> ' + (data.identification_type || 'No configurado') + '</li>';
                            debugHtml += '<li><strong>Número de identificación:</strong> ' + (data.identification_number || 'No configurado') + '</li>';
                            debugHtml += '<li><strong>Email:</strong> ' + (data.billing_email || 'No configurado') + '</li>';
                            debugHtml += '<li><strong>Teléfono:</strong> ' + (data.billing_phone || 'No configurado') + '</li>';
                            debugHtml += '<li><strong>Nombre completo:</strong> ' + (data.billing_first_name || '') + ' ' + (data.billing_last_name || '') + '</li>';
                            
                            if (data.use_company) {
                                debugHtml += '<li><strong>Empresa:</strong> ' + (data.company_name || 'No configurado') + '</li>';
                            }
                            
                            debugHtml += '</ul>';
                            
                            // Mostrar todos los metadatos
                            if (data.all_meta_data && Object.keys(data.all_meta_data).length > 0) {
                                debugHtml += '<p><strong>Todos los metadatos del pedido:</strong></p>';
                                debugHtml += '<pre style="background: #f9f9f9; padding: 10px; border-radius: 4px; font-size: 12px; max-height: 300px; overflow-y: auto;">';
                                debugHtml += JSON.stringify(data.all_meta_data, null, 2);
                                debugHtml += '</pre>';
                            }
                            
                            debugHtml += '</div>';
                            infoDiv.html(debugHtml).show();
                        } else {
                            infoDiv.html('<div class="notice notice-error"><p>Error al depurar pedido: ' + response.data + '</p></div>').show();
                        }
                    },
                    error: function() {
                        infoDiv.html('<div class="notice notice-error"><p>Error de conexión al depurar pedido.</p></div>').show();
                    },
                    complete: function() {
                        button.prop('disabled', false).text('Depurar pedido');
                    }
                });
            });
            
            // Inicializar carga automática después de un breve delay
            setTimeout(function() {
                autoLoadPosCatalog();
            }, 500);
        });
        </script>
        <?php
    }
    
    /**
     * AJAX: Probar conexión
     */
    public function ajax_test_connection() {
        check_ajax_referer('factunexo_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Acceso denegado');
        }
        
        $api = new Factunexo_API();
        $result = $api->test_connection();
        
        if (is_wp_error($result)) {
            // Agregar información de depuración al error
            $error_data = $result->get_error_data();
            $debug_info = array();
            
            if (isset($error_data['status'])) {
                $debug_info['http_code'] = $error_data['status'];
            }
            if (isset($error_data['url'])) {
                $debug_info['api_url'] = $error_data['url'];
            }
            if (isset($error_data['response_body'])) {
                $debug_info['response'] = $error_data['response_body'];
            }
            
            // Log del error para depuración
            error_log('Factunexo Connection Test Failed: ' . $result->get_error_message());
            if (!empty($debug_info)) {
                error_log('Debug Info: ' . print_r($debug_info, true));
            }
            
            wp_send_json_error($result->get_error_message());
        } else {
            // Log del éxito para depuración
            error_log('Factunexo Connection Test Success: ' . $result['message']);
            wp_send_json_success($result);
        }
    }
    
    /**
     * AJAX: Obtener catálogo de POS
     */
    public function ajax_get_pos_catalog() {
        check_ajax_referer('factunexo_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Acceso denegado');
        }
        
        $api = new Factunexo_API();
        $result = $api->get_pos_catalog();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            // El backend devuelve directamente la lista, no un objeto con 'content'
            wp_send_json_success($result);
        }
    }
    
    /**
     * AJAX: Verificar logs de error
     */
    public function ajax_check_error_logs() {
        check_ajax_referer('factunexo_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Acceso denegado');
        }
        
        // Obtener logs recientes de error de WordPress
        $log_file = ini_get('error_log');
        $logs = array();
        
        if ($log_file && file_exists($log_file)) {
            $log_content = file_get_contents($log_file);
            $log_lines = explode("\n", $log_content);
            
            // Filtrar solo logs relacionados con Factunexo de las últimas 24 horas
            $recent_logs = array();
            $one_day_ago = time() - (24 * 60 * 60);
            
            foreach ($log_lines as $line) {
                if (strpos($line, 'Factunexo') !== false) {
                    // Intentar extraer timestamp del log
                    if (preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\]/', $line, $matches)) {
                        $log_time = strtotime($matches[1]);
                        if ($log_time >= $one_day_ago) {
                            $recent_logs[] = $line;
                        }
                    }
                }
            }
            
            // Limitar a los últimos 10 logs
            $logs = array_slice($recent_logs, -10);
        }
        
        wp_send_json_success(array('logs' => $logs));
    }
    
    /**
     * AJAX: Probar conectividad básica
     */
    public function ajax_test_basic_connectivity() {
        check_ajax_referer('factunexo_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Acceso denegado');
        }
        
        $api = new Factunexo_API();
        $result = $api->test_basic_connectivity();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success($result);
        }
    }
    
    /**
     * AJAX: Depurar pedido
     */
    public function ajax_debug_order() {
        check_ajax_referer('factunexo_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Acceso denegado');
        }
        
        $order_id = intval($_POST['order_id']);
        
        if (!$order_id) {
            wp_send_json_error('ID de pedido inválido');
        }
        
        $order = wc_get_order($order_id);
        
        if (!$order) {
            wp_send_json_error('Pedido no encontrado');
        }
        
        $customer_service = new Factunexo_Customer();
        $debug_info = $customer_service->debug_customer_data($order);
        
        wp_send_json_success($debug_info);
    }
    
    /**
     * AJAX: Mostrar JSON de debug de orden
     */
    public function ajax_show_json_debug() {
        check_ajax_referer('factunexo_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Acceso denegado');
        }
        
        $order_id = intval($_POST['order_id']);
        
        if (!$order_id) {
            wp_send_json_error('ID de pedido requerido');
        }
        
        $order = wc_get_order($order_id);
        
        if (!$order) {
            wp_send_json_error('Pedido no encontrado');
        }
        
        // Obtener el JSON guardado en el campo personalizado
        $json_data = $order->get_meta('_factunexo_invoice_json');
        
        if (empty($json_data)) {
            wp_send_json_error('No se encontró JSON de debug para este pedido. El pedido puede no haber sido procesado por Factunexo.');
        }
        
        // Decodificar y formatear el JSON
        $decoded_json = json_decode($json_data, true);
        $formatted_json = json_encode($decoded_json, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        wp_send_json_success(array(
            'json' => $formatted_json,
            'order_id' => $order_id
        ));
    }
}
